<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class CPanelService
{
    protected $domain;
    protected $username;
    protected $apiToken;
    protected $baseUrl;

    public function __construct()
    {
        $domain = env('CPANEL_DOMAIN');
        // Strip https:// if it exists to prevent double https://
        $this->domain = preg_replace('/^https?:\/\//', '', $domain);
        $this->username = env('CPANEL_USERNAME');
        $this->apiToken = env('CPANEL_API_TOKEN');
        // We will use both UAPI and API2 by changing the endpoint dynamically in call()
        $this->baseUrl = "https://{$this->domain}:2083/json-api/cpanel";
    }

    // Build absolute home path
    protected function homePath($relativePath = '')
    {
        $base = '/home/' . $this->username;
        if (empty($relativePath) || $relativePath === '/')
            return $base;
        return $base . '/' . ltrim($relativePath, '/');
    }

    // Build path relative to home (e.g. public_html/file.txt)
    protected function relativeHomePath($path)
    {
        $homePrefix = '/home/' . $this->username;
        $rel = str_replace($homePrefix, '', $path);
        return ltrim($rel, '/');
    }

    /**
     * Convert app path to cPanel API path (relative to home).
     * - paths like "domains/example.com" stay as-is (folder outside public_html).
     * - other paths get public_html/ prefix (legacy addon inside public_html).
     */
    protected function toCpanelPath($path)
    {
        $path = ltrim((string) $path, '/');
        if ($path === '' || str_starts_with($path, 'public_html') || str_starts_with($path, 'domains/')) {
            return $path;
        }
        return 'public_html/' . $path;
    }

    public function callApi($module, $function, $params = [], $version = 'uapi')
    {
        return $this->call($module, $function, $params, $version);
    }

    protected function call($module, $function, $params = [], $version = 'uapi')
    {
        if ($version === 'uapi') {
            $url = "https://{$this->domain}:2083/execute/{$module}/{$function}";
            $response = Http::withHeaders([
                'Authorization' => "cpanel {$this->username}:{$this->apiToken}"
            ])
                ->withoutVerifying()
                ->get($url, $params);
        } else {
            // API2 requires specific query parameters for the base URL
            $url = "https://{$this->domain}:2083/json-api/cpanel";
            $api2Params = array_merge([
                'cpanel_jsonapi_user' => $this->username,
                'cpanel_jsonapi_apiversion' => 2,
                'cpanel_jsonapi_module' => $module,
                'cpanel_jsonapi_func' => $function,
            ], $params);

            $response = Http::withHeaders([
                'Authorization' => "cpanel {$this->username}:{$this->apiToken}"
            ])
                ->withoutVerifying()
                ->post($url, $api2Params); // Using POST for API2
        }

        $data = $response->json();

        // Log the response for debugging purposes
        \Log::info("cPanel API Call ({$version}): {$module}/{$function}", [
            'status' => $response->status(),
            'response' => $data
        ]);

        return $data;
    }

    public function getStats()
    {
        return $this->call('StatsBar', 'get_stats', [
            'display' => 'diskusage|bandwidthusage|emailaccounts|mysql_databases'
        ]);
    }

    /**
     * Returns current disk usage in GB (float).
     * Parses various cPanel StatsBar response formats robustly.
     */
    public function getDiskUsageGB(): float
    {
        try {
            $stats = $this->getStats();

            // StatsBar returns items keyed by stat name
            $disk = null;
            if (isset($stats['data']) && is_array($stats['data'])) {
                foreach ($stats['data'] as $item) {
                    if (isset($item['name']) && $item['name'] === 'diskusage') {
                        $disk = $item;
                        break;
                    }
                }
            }
            if (!$disk) {
                $disk = $stats['diskusage'] ?? null;
            }

            if (!$disk) return 0.0;

            // _count is usually in MB, value may be "1.23 GB" or "512 MB"
            $count = (float) ($disk['_count'] ?? $disk['count'] ?? 0);
            $units = strtoupper(trim($disk['units'] ?? 'MB'));

            return match ($units) {
                'GB'  => $count,
                'KB'  => $count / 1024 / 1024,
                'B'   => $count / 1024 / 1024 / 1024,
                default => $count / 1024, // MB → GB
            };
        } catch (\Throwable) {
            return 0.0;
        }
    }

    /**
     * Returns recursive disk usage for a specific directory (relative to home), in GB.
     * Used for per-user/domain storage calculations instead of account-wide stats.
     */
    public function getDirectoryUsageGB(string $relativePath): float
    {
        try {
            $relativePath = ltrim(str_replace('\\', '/', $relativePath), '/');
            if ($relativePath === '') return 0.0;

            // Prefer home-relative path (same style as other Fileman calls)
            $dirParam = $this->toCpanelPath($relativePath); // e.g. domains/example.com

            $res = $this->call('Fileman', 'getdiskusage', [
                'dir'             => $dirParam,
                'show_hidden'     => 1,
                'all'             => 1,
                'calculate_du'    => 1,
                'need_bytes'      => 1,
                'need_dir_sizes'  => 0,
                'need_file_count' => 0,
            ]);

            // Try common shapes
            $bytes = null;
            if (isset($res['diskusage']['total_bytes'])) {
                $bytes = (float) $res['diskusage']['total_bytes'];
            } elseif (isset($res['data'][0]['total_bytes'])) {
                $bytes = (float) $res['data'][0]['total_bytes'];
            } elseif (isset($res['diskusage']['bytes'])) {
                $bytes = (float) $res['diskusage']['bytes'];
            }

            // If first attempt failed, try again with absolute path (fallback)
            if ((!$bytes || $bytes < 0)) {
                $abs = $this->homePath($dirParam);
                $res2 = $this->call('Fileman', 'getdiskusage', [
                    'dir'             => $abs,
                    'show_hidden'     => 1,
                    'all'             => 1,
                    'calculate_du'    => 1,
                    'need_bytes'      => 1,
                    'need_dir_sizes'  => 0,
                    'need_file_count' => 0,
                ]);

                if (isset($res2['diskusage']['total_bytes'])) {
                    $bytes = (float) $res2['diskusage']['total_bytes'];
                } elseif (isset($res2['data'][0]['total_bytes'])) {
                    $bytes = (float) $res2['data'][0]['total_bytes'];
                } elseif (isset($res2['diskusage']['bytes'])) {
                    $bytes = (float) $res2['diskusage']['bytes'];
                }
            }

            if (!$bytes || $bytes < 0) return 0.0;

            return $bytes / 1024 / 1024 / 1024; // bytes → GB
        } catch (\Throwable) {
            return 0.0;
        }
    }

    /**
     * Recursively walks a directory (relative to home) and sums file sizes using Fileman::list_files.
     * Fallback when getdiskusage is unreliable on some cPanel setups.
     */
    public function getDirectoryUsageGBRecursive(string $relativePath): float
    {
        $relativePath = trim(str_replace('\\', '/', $relativePath), '/');
        if ($relativePath === '') return 0.0;

        $totalBytes = 0;
        $queue = [$relativePath];

        while (!empty($queue)) {
            $dir = array_pop($queue);
            try {
                $res   = $this->listFiles($dir);
                $items = $res['data'] ?? [];
            } catch (\Throwable) {
                continue;
            }

            foreach ($items as $item) {
                $type = $item['type'] ?? '';
                $name = $item['file'] ?? $item['name'] ?? '';
                if ($name === '' || $name === '.' || $name === '..') continue;

                if ($type === 'dir') {
                    $queue[] = trim($dir . '/' . $name, '/');
                } else {
                    $size = $item['size'] ?? 0;
                    $totalBytes += (int) $size;
                }
            }
        }

        return $totalBytes / 1024 / 1024 / 1024;
    }

    public function getEmailAccounts()
    {
        return $this->call('Email', 'list_pop_with_disk');
    }

    public function getDomains()
    {
        // 1. Get UAPI data (usually contains Main domain)
        $uapi = $this->call('DomainInfo', 'list_domains');

        // 2. Get API2 data (specifically for Addon domains)
        $api2 = $this->call('AddonDomain', 'listaddondomains', [], 'api2');

        // Merge them into one structure for the view to parse
        return [
            'uapi' => $uapi,
            'api2' => $api2
        ];
    }


    public function getDatabases()
    {
        return $this->call('Mysql', 'list_databases');
    }

    /**
     * Add addon domain with document root OUTSIDE public_html: domains/DOMAIN (e.g. domains/example.com).
     * One domain = one folder under home, separate from public_html.
     */
        // Using API2 as it's more stable for domain creation on older/specific cPanel setups
    public function addAddonDomain($newDomain, $subdomain, $dir)
    {
        return $this->call('AddonDomain', 'addaddondomain', [
            'newdomain' => $newDomain,
            'subdomain' => $subdomain,
            'dir' => $dir,
        ], 'api2');
    }

    public function deleteDomain($domain, $subdomain = null)
    {
        // 1. Try UAPI first (Modern)
        $uapiAttempts = ['delete_addon', 'deleteaddon', 'del_addon'];
        foreach ($uapiAttempts as $func) {
            $res = $this->call('AddonDomain', $func, ['domain' => $domain], 'uapi');
            if (isset($res['status']) && $res['status'] == 1) {
                return $res;
            }
        }

        // 2. Try API2 with provided subdomain
        $sub = $subdomain ?? explode('.', $domain)[0];
        $res1 = $this->call('AddonDomain', 'deladdondomain', [
            'domain' => $domain,
            'subdomain' => $sub,
            'skip_files' => 0
        ], 'api2');

        // Check if API2 worked
        if (isset($res1['cpanelresult']['data'][0]['result']) && $res1['cpanelresult']['data'][0]['result'] == 1) {
            return $res1;
        }

        // 3. SPECIAL FALLBACK: If "does not correspond" error occurs, 
        // try variations of the subdomain name
        $variations = [
            $domain, // Some cPanel versions store subdomain as the full domain
            str_replace('.', '_', $domain), // Replace dots with underscores
            $sub . '.' . $this->domain // Full subdomain name
        ];

        foreach ($variations as $v) {
            if ($v == $sub)
                continue; // Skip if same as first attempt

            $resV = $this->call('AddonDomain', 'deladdondomain', [
                'domain' => $domain,
                'subdomain' => $v,
                'skip_files' => 0
            ], 'api2');

            if (isset($resV['cpanelresult']['data'][0]['result']) && $resV['cpanelresult']['data'][0]['result'] == 1) {
                return $resV;
            }
        }

        // Return the first API2 response if everything fails
        return $res1;
    }

    public function listFiles($dir)
    {
        return $this->call('Fileman', 'list_files', [
            'dir' => $this->toCpanelPath($dir),
            'show_hidden' => 1
        ]);
    }

    /**
     * Collect all files under a directory recursively. Returns flat list of [dir => path, file => name].
     * Used for building flat zip (all files in zip root).
     */
    public function collectFilesRecursive($dir)
    {
        $raw = $this->listFiles($dir);
        $data = $raw['data'] ?? $raw['result']['data'] ?? $raw['cpanelresult']['data'] ?? [];
        if (!is_array($data)) {
            $data = [];
        }
        $out = [];
        foreach ($data as $f) {
            $name = $f['file'] ?? $f['name'] ?? '';
            $type = $f['type'] ?? '';
            if ($name === '' || $name === '..' || $name === '.') {
                continue;
            }
            if ($type === 'dir') {
                $subDir = $dir . '/' . $name;
                $out = array_merge($out, $this->collectFilesRecursive($subDir));
            } else {
                $out[] = ['dir' => $dir, 'file' => $name];
            }
        }
        return $out;
    }
        // API2 Fileman::fileop for delete - use home-relative path

    public function deleteFile($fullPath)
    {
        return $this->call('Fileman', 'fileop', [
            'op' => 'unlink',
            'sourcefiles' => $this->toCpanelPath($fullPath),
            'doubledecode' => 0
        ], 'api2');
    }

    public function uploadFile($dir, $fileObject)
    {
        $url = "https://{$this->domain}:2083/execute/Fileman/upload_files";

        $response = Http::withHeaders([
            'Authorization' => "cpanel {$this->username}:{$this->apiToken}"
        ])
            ->withoutVerifying()
            ->attach('file-0', file_get_contents($fileObject->getRealPath()), $fileObject->getClientOriginalName())
            ->post($url, [
                'dir' => $this->toCpanelPath($dir),
                'overwrite' => 1
            ]);

        return $response->json();
    }

    public function createFolder($dir, $folderName)
    {
        // Neither UAPI mkdir nor API2 mkdir work for domains/ paths.
        // Strategy: build a tiny zip locally with a placeholder file, upload it,
        // then extract directly into the new folder path (cPanel creates the destination dir).
        // Finally clean up the zip and placeholder.

        $relDir    = $this->toCpanelPath(ltrim($dir, '/'));        // e.g. domains/11111111111.com
        $zipName   = '_mkdir_' . uniqid() . '.zip';
        $tmpZip    = sys_get_temp_dir() . DIRECTORY_SEPARATOR . $zipName;

        // 1. Build zip containing one empty placeholder file
        $zip = new \ZipArchive();
        if ($zip->open($tmpZip, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== true) {
            return ['status' => 0, 'errors' => ['Could not create temporary zip.']];
        }
        $zip->addFromString('.keep', '');
        $zip->close();

        // 2. Upload the zip to $dir on cPanel
        \Log::info('cPanel createFolder: uploading zip', ['dir' => $relDir, 'zip' => $zipName]);
        $uploadUrl = "https://{$this->domain}:2083/execute/Fileman/upload_files";
        $uploadRes = Http::withHeaders(['Authorization' => "cpanel {$this->username}:{$this->apiToken}"])
            ->withoutVerifying()
            ->attach('file-0', file_get_contents($tmpZip), $zipName)
            ->post($uploadUrl, ['dir' => $relDir, 'overwrite' => 1]);
        @unlink($tmpZip);

        $uploadData = $uploadRes->json();
        \Log::info('cPanel createFolder: upload response', ['res' => $uploadData]);
        $uploadStatus = $uploadData['status'] ?? $uploadData['result']['status'] ?? null;
        if ($uploadStatus != 1) {
            $err = $uploadData['errors'][0] ?? $uploadData['result']['errors'][0] ?? 'Zip upload failed.';
            return ['status' => 0, 'errors' => [$err]];
        }

        // 3. Extract the zip into the NEW folder path (absolute) — cPanel creates the dir
        $srcAbs = $this->homePath($relDir . '/' . $zipName);
        $dstAbs = $this->homePath($relDir . '/' . $folderName);
        \Log::info('cPanel createFolder: extracting into new dir', ['src' => $srcAbs, 'dst' => $dstAbs]);
        $extractRes = $this->call('Fileman', 'fileop', [
            'op'          => 'extract',
            'sourcefiles' => $srcAbs,
            'destfiles'   => $dstAbs,
            'doubledecode' => 0,
        ], 'api2');
        \Log::info('cPanel createFolder: extract response', ['res' => $extractRes]);

        // 4. Clean up — use home-relative paths (toCpanelPath handles domains/ correctly)
        $this->deleteFile($relDir . '/' . $zipName);
        $this->deleteFile($relDir . '/' . $folderName . '/.keep');

        $data  = $extractRes['cpanelresult']['data'][0] ?? null;
        $event = $extractRes['cpanelresult']['event']['result'] ?? 1;
        if ($data && isset($data['result']) && (int)$data['result'] === 0) {
            $msg = $data['reason'] ?? $data['output'] ?? 'Folder creation failed.';
            return ['status' => 0, 'errors' => [$msg]];
        }

        return ['status' => 1];
    }

    /**
     * Extract zip using cPanel API2 Fileman::fileop (op=extract).
     * After cPanel extracts, any new subfolder is flattened: its files are moved into $dir directly.
     */
    public function extractFile($dir, $fileName)
    {
        $dir = ltrim(str_replace('\\', '/', (string)$dir), '/');

        // Snapshot directory contents before extraction to detect new items
        $beforeRaw  = $this->listFiles($dir);
        $beforeData = $beforeRaw['data'] ?? $beforeRaw['result']['data'] ?? $beforeRaw['cpanelresult']['data'] ?? [];
        if (!is_array($beforeData)) $beforeData = [];
        $beforeNames = [];
        foreach ($beforeData as $f) {
            $n = $f['file'] ?? $f['name'] ?? '';
            if ($n !== '' && $n !== '.' && $n !== '..') $beforeNames[$n] = true;
        }

        // Use absolute paths so cPanel extracts to the exact directory (not relative to zip location)
        $src = $this->homePath($dir . '/' . $fileName); // e.g. /home/user/domains/example.com/file.zip
        $dst = $this->homePath($dir);                   // e.g. /home/user/domains/example.com
        \Log::info('cPanel API2 extract', ['src' => $src, 'dst' => $dst]);

        $res      = $this->call('Fileman', 'fileop', [
            'op'           => 'extract',
            'sourcefiles'  => $src,
            'destfiles'    => $dst,
            'doubledecode' => 0,
        ], 'api2');

        \Log::info('API2 extract response', ['res' => $res]);

        $cpResult    = $res['cpanelresult'] ?? [];
        $data        = $cpResult['data'][0] ?? null;
        $eventResult = $cpResult['event']['result'] ?? null;

        // Explicit API failure
        if ($data && isset($data['result']) && (int)$data['result'] === 0) {
            $msg = $data['reason'] ?? $data['output'] ?? 'Extract failed.';
            return ['errors' => [$msg]];
        }

        // Snapshot after extraction
        $afterRaw  = $this->listFiles($dir);
        $afterData = $afterRaw['data'] ?? $afterRaw['result']['data'] ?? $afterRaw['cpanelresult']['data'] ?? [];
        if (!is_array($afterData)) $afterData = [];

        $newDirs      = [];
        $newFileCount = 0;
        foreach ($afterData as $f) {
            $n    = $f['file'] ?? $f['name'] ?? '';
            $type = $f['type'] ?? '';
            if ($n === '' || $n === '.' || $n === '..' || isset($beforeNames[$n])) continue;
            if ($type === 'dir') $newDirs[] = $n;
            else $newFileCount++;
        }

        // Nothing appeared — extract silently failed (e.g. path restricted by cPanel)
        if (empty($newDirs) && $newFileCount === 0) {
            \Log::warning('API2 extract: no new items appeared', ['dir' => $dir, 'file' => $fileName, 'event' => $eventResult, 'data' => $data]);
            return ['errors' => ['Extract produced no files. The archive may be empty or the server rejected the path.']];
        }

        // Flatten: move files from any new subfolder(s) directly into $dir
        foreach ($newDirs as $subName) {
            $subDir = $dir . '/' . $subName;
            $files  = $this->collectFilesRecursive($subDir);
            foreach ($files as $entry) {
                try {
                    $this->moveFile($entry['dir'], $entry['file'], $dir);
                } catch (\Throwable $e) {
                    \Log::warning('Extract flatten: move failed', ['entry' => $entry, 'err' => $e->getMessage()]);
                }
            }
            $this->deleteFolderRecursive($subDir);
        }

        return ['status' => 1, 'result' => ['Extracted successfully.']];
    }

    /** @deprecated kept for reference only */
    protected function extractFileViaCpanelThenFlatten($dir, $fileName)
    {
        return $this->extractFile($dir, $fileName);
    }

    /**
     * Delete a directory and its contents (files then dirs).
     */
    protected function deleteFolderRecursive($dirPath)
    {
        $raw = $this->listFiles($dirPath);
        $data = $raw['data'] ?? $raw['result']['data'] ?? $raw['cpanelresult']['data'] ?? [];
        if (!is_array($data)) {
            $data = [];
        }
        foreach ($data as $f) {
            $name = $f['file'] ?? $f['name'] ?? '';
            $type = $f['type'] ?? '';
            if ($name === '' || $name === '.' || $name === '..') {
                continue;
            }
            $fullPath = $dirPath . '/' . $name;
            if ($type === 'dir') {
                $this->deleteFolderRecursive($fullPath);
            } else {
                $this->deleteFile($fullPath);
            }
        }
        $this->deleteFile(rtrim($dirPath, '/'));
    }

    public function downloadFile($dir, $fileName)
    {
        // Path relative to home (same format as list_files) so domains/ and public_html/ both work
        $dirParam = $this->toCpanelPath(ltrim(str_replace('\\', '/', (string) $dir), '/'));
        $response = $this->call('Fileman', 'get_file_content', [
            'dir' => $dirParam,
            'file' => $fileName
        ]);
        return $response;
    }


    /**
     * Get raw file content as string. Decodes base64 if returned by API.
     * Handles both top-level and UAPI result.result response structure.
     */
    public function getFileContent($dir, $fileName)
    {
        $result = $this->downloadFile($dir, $fileName);
        $status = $result['status'] ?? $result['result']['status'] ?? null;
        $data = $result['data'] ?? $result['result']['data'] ?? null;
        if ($status != 1 || !is_array($data)) {
            return null;
        }
        $content = $data['content'] ?? '';
        if (isset($data['encoding']) && $data['encoding'] === 'base64') {
            $content = base64_decode($content, true);
            if ($content === false) {
                return null;
            }
        }
        return $content;
    }

    public function saveFileContent($dir, $fileName, $content)
    {
        return $this->call('Fileman', 'save_file_content', [
            'dir' => $this->toCpanelPath($dir),
            'file' => $fileName,
            'content' => $content
        ]);
    }

    public function bulkDelete($dir, $filesArray)
    {
        $results = [];
        foreach ($filesArray as $file) {
            $fullPath = $dir . '/' . $file;
            $results[] = $this->deleteFile($fullPath);
        }
        return $results;
    }

    // Create an empty file
    public function createFile($dir, $fileName, $content = '')
    {
        return $this->call('Fileman', 'save_file_content', [
            'dir' => $this->toCpanelPath($dir),
            'file' => $fileName,
            'content' => $content
    // Copy files - home-relative paths
        ]);
    }

    public function copyFile($sourceDir, $fileName, $destDir)
    {
        // Use absolute paths to avoid cPanel treating them as relative (path-doubling bug)
        $src = $this->homePath(ltrim($this->toCpanelPath(ltrim($sourceDir, '/') . '/' . $fileName), '/'));
        $dst = $this->homePath(ltrim($this->toCpanelPath(ltrim($destDir, '/') . '/' . $fileName), '/'));
        \Log::info('cPanel copyFile', ['src' => $src, 'dst' => $dst]);
        return $this->call('Fileman', 'fileop', [
            'op'           => 'copy',
            'sourcefiles'  => $src,
            'destfiles'    => $dst,
            'doubledecode' => 0
        ], 'api2');
    }

    public function moveFile($sourceDir, $fileName, $destDir)
    {
        // Use absolute paths to avoid cPanel treating them as relative (path-doubling bug)
        $src = $this->homePath(ltrim($this->toCpanelPath(ltrim($sourceDir, '/') . '/' . $fileName), '/'));
        $dst = $this->homePath(ltrim($this->toCpanelPath(ltrim($destDir, '/') . '/' . $fileName), '/'));
        \Log::info('cPanel moveFile', ['src' => $src, 'dst' => $dst]);
        return $this->call('Fileman', 'fileop', [
            'op'           => 'move',
            'sourcefiles'  => $src,
            'destfiles'    => $dst,
            'doubledecode' => 0
        ], 'api2');
    }

    public function renameFile($dir, $oldName, $newName)
    {
        $base = ltrim($this->toCpanelPath(ltrim($dir, '/')), '/');
        return $this->call('Fileman', 'fileop', [
            'op'          => 'rename',
            'sourcefiles' => $this->homePath($base . '/' . $oldName),
            'destfiles'   => $this->homePath($base . '/' . $newName),
            'doubledecode' => 0
        ], 'api2');
    }

    /**
     * Final fix for compressFiles:
     * 1. Uses pipe (|) instead of comma to avoid split issues
     * 2. Uses arg-0, type, and metadata to force 'zip'
     * 3. Uses home-relative paths WITHOUT leading slash
     */
    public function compressFiles($dir, $files, $destFile)
    {
        if (!preg_match('/\.(zip|tar\.gz|tgz|tar\.bz2|tar|gz|bz2)$/i', $destFile)) {
            $destFile .= '.zip';
        }

        $relDir = $this->toCpanelPath($dir);
        $sourceList = is_array($files)
            ? implode('|', array_map(fn($f) => ($relDir ? $relDir . '/' : '') . $f, $files))
            : ($relDir ? $relDir . '/' : '') . $files;

        $destProp = ($relDir ? $relDir . '/' : '') . $destFile;

        $params = [
            'op' => 'compress',
            'arg-0' => 'zip',
            'type' => 'zip',
            'metadata' => 'zip',
            'sourcefiles' => $sourceList,
            'destfiles' => $destProp,
            'doubledecode' => 0
        ];

        \Log::info('cPanel API2 Compress (Deep Fix)', $params);

    // Change file/folder permissions - home-relative path
        return $this->call('Fileman', 'fileop', $params, 'api2');
    }

    public function changePermissions($dir, $fileName, $perms)
    {
        return $this->call('Fileman', 'fileop', [
            'op' => 'chmod',
            'sourcefiles' => $this->toCpanelPath($dir . '/' . $fileName),
            'metadata' => $perms,
            'doubledecode' => 0
        ], 'api2');
    }

    // ==================== DATABASE (MySQL via UAPI) ====================

    public function listDatabases()
    {
        return $this->call('Mysql', 'list_databases');
    }

    public function createDatabase(string $dbName)
    {
        return $this->call('Mysql', 'create_database', ['name' => $dbName]);
    }

    public function deleteDatabase(string $dbName)
    {
        return $this->call('Mysql', 'delete_database', ['name' => $dbName]);
    }

    public function listDbUsers()
    {
        return $this->call('Mysql', 'list_users');
    }

    public function createDbUser(string $username, string $password)
    {
        return $this->call('Mysql', 'create_user', [
            'name'     => $username,
            'password' => $password,
        ]);
    }

    public function deleteDbUser(string $username)
    {
        return $this->call('Mysql', 'delete_user', ['name' => $username]);
    }

    public function assignUserToDb(string $username, string $dbName, string $type = 'all')
    {
        // privileges: ALL PRIVILEGES | READ | WRITE
        $privMap = ['all' => 'ALL PRIVILEGES', 'rw' => 'ALL PRIVILEGES', 'ro' => 'READ'];
        return $this->call('Mysql', 'set_privileges_on_database', [
            'user'       => $username,
            'database'   => $dbName,
            'privileges' => $privMap[$type] ?? 'ALL PRIVILEGES',
        ]);
    }

    public function revokeUserFromDb(string $username, string $dbName)
    {
        return $this->call('Mysql', 'remove_user_from_database', [
            'user'     => $username,
            'database' => $dbName,
        ]);
    }

    public function changeDbUserPassword(string $username, string $password)
    {
        return $this->call('Mysql', 'change_password', [
            'name'     => $username,
            'password' => $password,
        ]);
    }

    /** Returns db→users privileges map */
    public function listDbPrivileges()
    {
        return $this->call('Mysql', 'get_privileges_on_database');
    }

    public function getDbPrefix(): string
    {
        return $this->username . '_';
    }

    // ==================== SUBDOMAINS ====================

    public function listSubdomains()
    {
        return $this->call('SubDomain', 'listsubdomains');
    }

    public function createSubdomain(string $subdomain, string $rootDomain, string $dir = '')
    {
        // Keep outside public_html — mirror addon domains:
        // subdomain of example.com → domains/example.com/blog
        $cleanSub  = trim($subdomain);
        $cleanRoot = trim($rootDomain);
        $docRoot   = $dir ?: "domains/{$cleanRoot}/{$cleanSub}";

        return $this->call('SubDomain', 'addsubdomain', [
            'domain'      => $cleanSub,
            'rootdomain'  => $cleanRoot,
            'dir'         => $docRoot,
            'disallowdot' => 0,
        ]);
    }

    public function deleteSubdomain(string $subdomain)
    {
        // UAPI does not support delsubdomain — must use API2
        // $subdomain must be the full hostname e.g. "blog.example.com"
        return $this->call('SubDomain', 'delsubdomain', [
            'domain' => $subdomain,
        ], 'api2');
    }

    // ==================== DNS RECORDS ====================

    public function getDnsZone(string $domain)
    {
        return $this->call('DNS', 'parse_zone', ['zone' => $domain]);
    }

    public function addDnsRecord(string $domain, array $record)
    {
        return $this->call('DNS', 'add_zone_record', array_merge(['zone' => $domain], $record));
    }

    public function editDnsRecord(string $domain, int $line, array $record)
    {
        return $this->call('DNS', 'edit_zone_record', array_merge(['zone' => $domain, 'line' => $line], $record));
    }

    public function removeDnsRecord(string $domain, int $line)
    {
        return $this->call('DNS', 'remove_zone_record', ['zone' => $domain, 'line' => $line]);
    }

    // ==================== EMAIL ====================

    public function listEmailAccounts(string $domain = '')
    {
        $params = ['regex' => ''];
        if ($domain) $params['domain'] = $domain;
        return $this->call('Email', 'list_pops_with_disk', $params);
    }

    public function createEmailAccount(string $email, string $password, string $domain, int $quotaMb = 250)
    {
        return $this->call('Email', 'add_pop', [
            'email'    => $email,
            'password' => $password,
            'domain'   => $domain,
            'quota'    => $quotaMb,
        ]);
    }

    public function deleteEmailAccount(string $email, string $domain)
    {
        return $this->call('Email', 'delete_pop', [
            'email'  => $email,
            'domain' => $domain,
        ]);
    }

    public function changeEmailPassword(string $email, string $domain, string $password)
    {
        return $this->call('Email', 'passwd_pop', [
            'email'    => $email,
            'domain'   => $domain,
            'password' => $password,
        ]);
    }

    public function getEmailDiskUsage(string $email, string $domain)
    {
        return $this->call('Email', 'get_pop_quota', [
            'account' => $email,
            'domain'  => $domain,
        ]);
    }

    // ==================== PHP VERSION ====================

    public function getPhpInstalledVersions(): array
    {
        // Try UAPI first
        $res  = $this->call('LangPHP', 'php_get_installed_versions');
        $data = $this->extractPhpData($res);
        if (!empty($data)) return $data;

        // Fallback: try API2 MultiPHP (older cPanel)
        $res2  = $this->call('MultiPHP', 'php_get_installed_versions', [], 'api2');
        $data2 = $this->extractPhpData($res2, true);
        if (!empty($data2)) return $data2;

        return [];
    }

    public function getAllDomainPhpVersions(): array
    {
        $res  = $this->call('LangPHP', 'php_get_vhost_versions');
        $data = $res['data'] ?? null;
        if (is_array($data) && !empty($data)) return $data;

        // Fallback: API2
        $res2  = $this->call('LangPHP', 'php_get_vhost_versions', [], 'api2');
        $data2 = $res2['cpanelresult']['data'] ?? [];
        return is_array($data2) ? $data2 : [];
    }

    /** Extract version strings from UAPI or API2 response */
    private function extractPhpData(array $res, bool $isApi2 = false): array
    {
        // UAPI: { data: [...] }
        if (!$isApi2) {
            $data = $res['data'] ?? null;
            if (is_array($data) && !empty($data)) return $data;
        }
        // API2: { cpanelresult: { data: [...] } }
        $data = $res['cpanelresult']['data'] ?? null;
        if (is_array($data) && !empty($data)) return $data;

        return [];
    }

    public function setDomainPhpVersion(string $domain, string $version): array
    {
        return $this->call('LangPHP', 'php_set_vhost_versions', [
            'version'  => $version,
            'vhost-0'  => $domain,
        ]);
    }

    // ==================== SOFTACULOUS / SESSION ====================

    /**
     * Get a browser-usable cPanel URL for the given path.
     * Uses Session::create_temp_user_session (cPanel 58+) for proper SSO,
     * falls back to POST /login/ with API token.
     */
    public function getCpanelBrowserUrl(string $gotoPath = '/'): ?string
    {
        $host = preg_replace('#^https?://#', '', rtrim($this->domain, '/'));

        // ── Method 1: UAPI Session::create_temp_user_session ──────────
        try {
            $res = $this->call('Session', 'create_temp_user_session', ['service' => 'cpaneld']);
            $url = $res['data']['url'] ?? null;
            if ($url) {
                // Extract the cpsess part and build our target URL
                if (preg_match('#/(cpsess\w+)/#', $url, $m)) {
                    return "https://{$host}:2083/{$m[1]}{$gotoPath}";
                }
                return $url;
            }
        } catch (\Throwable) {}

        // ── Method 2: POST /login/ with API token ─────────────────────
        try {
            $response = \Illuminate\Support\Facades\Http::withoutVerifying()
                ->withOptions(['allow_redirects' => false])
                ->asForm()
                ->post("https://{$host}:2083/login/", [
                    'user'      => $this->username,
                    'api_token' => $this->apiToken,
                    'goto_uri'  => $gotoPath,
                ]);

            $location = $response->header('Location');
            if ($location && preg_match('#/(cpsess\w+)/#', $location, $m)) {
                return "https://{$host}:2083/{$m[1]}{$gotoPath}";
            }
        } catch (\Throwable) {}

        return null;
    }

    // Keep old method as alias for backward compat
    public function getCpanelSession(): ?array
    {
        $host = preg_replace('#^https?://#', '', rtrim($this->domain, '/'));
        $url  = $this->getCpanelBrowserUrl('/');
        if (!$url) return null;
        if (!preg_match('#/(cpsess\w+)/#', $url, $m)) return null;
        $theme = str_contains($url, 'paper_lantern') ? 'paper_lantern' : 'jupiter';
        return ['token' => $m[1], 'theme' => $theme, 'host' => $host];
    }

    // ==================== SSL / TLS ====================

    public function getSslForDomain(string $domain): array
    {
        $res = $this->call('SSL', 'fetch_best_for_domain', ['domain' => $domain]);
        return $res['data'] ?? $res;
    }

    public function listInstalledSsl(): array
    {
        $res = $this->call('SSL', 'list_certs');
        return $res['data'] ?? [];
    }

    public function getAutoSslMetadata(): array
    {
        $res = $this->call('Autossl', 'get_user_domains_metadata');
        return $res['data'] ?? [];
    }

    public function runAutoSsl(): array
    {
        return $this->call('Autossl', 'start_autossl_check');
    }

    public function installLetEncrypt(string $domain): array
    {
        // Queue domain for AutoSSL (Let's Encrypt)
        return $this->call('SSL', 'set_primary_ssl', ['domain' => $domain]);
    }

    public function removeSsl(string $domain): array
    {
        return $this->call('SSL', 'delete_ssl', ['domain' => $domain]);
    }

    public function checkSslStatus(string $domain): array
    {
        $res = $this->call('SSL', 'installed_host', ['domain' => $domain]);
        return $res['data'] ?? [];
    }

    // ==================== Domain Redirect ====================

    /**
     * Add a 302 redirect for a domain → renewUrl in cPanel.
     * Places an .htaccess rule in the domain's document root.
     */
    public function addDomainRedirect(string $docRoot, string $renewUrl): bool
    {
        // Write .htaccess with redirect rule
        $htaccess = "# SERVER-PANEL-RENEW-REDIRECT\nRewriteEngine On\nRewriteRule ^(.*)$ {$renewUrl} [R=302,L]\n";
        try {
            $this->saveFileContent($docRoot, '.htaccess', $htaccess);
            return true;
        } catch (\Throwable) {
            return false;
        }
    }

    /**
     * Remove the renewal redirect from a domain's .htaccess.
     */
    public function removeDomainRedirect(string $docRoot): bool
    {
        try {
            $res = $this->getFileContent($docRoot, '.htaccess');
            $content = $res['data']['content'] ?? '';
            // Strip our redirect block
            $cleaned = preg_replace('/# SERVER-PANEL-RENEW-REDIRECT\n.*\n.*\n/m', '', $content);
            $this->saveFileContent($docRoot, '.htaccess', $cleaned);
            return true;
        } catch (\Throwable) {
            return false;
        }
    }
}
