<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use App\Models\SupportMessage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'status',
        'plan_name',
        'billing_amount',
        'subscription_expires_at',
        'disk_quota',
        'disk_quota_unit',
        'renew_url',
        'redirect_applied',
        'domain_limit',
        'email_limit',
        'primary_domain',
    ];

    public function isAdmin()
    {
        return $this->role === 'admin';
    }

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at'       => 'datetime',
            'password'                => 'hashed',
            'subscription_expires_at' => 'datetime',
            'billing_amount'          => 'decimal:2',
        ];
    }

    /**
     * Returns disk quota in GB regardless of unit stored.
     */
    public function diskQuotaGB(): float
    {
        $val  = (float) ($this->disk_quota ?? 0);
        $unit = strtoupper($this->disk_quota_unit ?? 'GB');
        return match ($unit) {
            'KB'  => $val / (1024 * 1024),
            'MB'  => $val / 1024,
            'TB'  => $val * 1024,
            default => $val, // GB
        };
    }

    /**
     * Returns days until subscription expires (null = no expiry).
     */
    public function daysUntilExpiry(): ?int
    {
        if (!$this->subscription_expires_at) return null;
        return (int) now()->diffInDays($this->subscription_expires_at, false);
    }

    public function isSubscriptionActive(): bool
    {
        if ($this->status === 'suspended') return false;
        if ($this->subscription_expires_at === null) return true; // no expiry set = unlimited
        return $this->subscription_expires_at->isFuture();
    }

    public function subscriptionStatus(): string
    {
        if ($this->status === 'suspended') return 'suspended';
        if ($this->subscription_expires_at && $this->subscription_expires_at->isPast()) return 'expired';
        return 'active';
    }

    public function supportMessages()
    {
        return $this->hasMany(SupportMessage::class);
    }

    public function domains()
    {
        return $this->hasMany(UserDomain::class);
    }

    public function userDatabases()
    {
        return $this->hasMany(UserDatabase::class);
    }

    public function userDbUsers()
    {
        return $this->hasMany(UserDbUser::class);
    }
}
