<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Services\CPanelService;
use App\Models\UserDomain;

class SoftaculousController extends Controller
{
    protected CPanelService $cpanel;
    protected string $cpanelHost;

    public function __construct()
    {
        $this->cpanel = new CPanelService(
            env('CPANEL_DOMAIN'),
            env('CPANEL_USERNAME'),
            env('CPANEL_API_TOKEN')
        );
        $this->cpanelHost = preg_replace('#^https?://#', '', rtrim(env('CPANEL_DOMAIN', ''), '/'));
    }

    private function userDomains(): array
    {
        $user    = Auth::user();
        $domains = $user->primary_domain ? [$user->primary_domain] : [];
        $addon   = UserDomain::where('user_id', $user->id)->pluck('domain')->toArray();
        return array_unique(array_merge($domains, $addon));
    }

    // ── Main page ──────────────────────────────────────────────

    public function index()
    {
        $userDomains = $this->userDomains();

        // Popular apps catalogue (Softaculous soft IDs)
        $apps = self::appCatalogue();

        return view('dashboard.softaculous', compact('userDomains', 'apps'));
    }

    // ── Server-side SSO redirect → Softaculous ────────────────
    // GET /softaculous/go?soft_id=26&domain=example.com&dir=
    // This is opened directly in a new tab — no AJAX needed.

    public function go(Request $request)
    {
        $request->validate([
            'soft_id' => 'required|integer|min:1',
            'domain'  => 'required|string',
            'dir'     => 'sometimes|nullable|string|max:100',
        ]);

        $userDomains = $this->userDomains();
        if (!in_array($request->domain, $userDomains, true)) {
            abort(403, 'You do not own this domain.');
        }

        $softId = (int) $request->soft_id;
        $domain = $request->domain;
        $dir    = trim($request->dir ?? '');

        $params = http_build_query(array_filter([
            'act'    => 'software',
            'soft'   => $softId,
            'domain' => $domain,
            'dir'    => $dir,
        ]));

        $gotoPath = '/softaculous/?' . $params;

        // Try SSO — get a proper browser-usable cPanel URL
        $url = null;
        try {
            $url = $this->cpanel->getCpanelBrowserUrl($gotoPath);
        } catch (\Throwable $e) {
            \Log::warning('Softaculous SSO failed: ' . $e->getMessage());
        }

        if ($url) {
            return redirect()->away($url);
        }

        // Fallback: direct Softaculous URL (works if already logged in to cPanel)
        return view('dashboard.softaculous_loader', [
            'cpanelHost' => $this->cpanelHost,
            'gotoPath'   => $gotoPath,
            'softId'     => $softId,
            'domain'     => $domain,
        ]);
    }

    // ── App catalogue ──────────────────────────────────────────

    public static function appCatalogue(): array
    {
        return [
            'CMS' => [
                ['id' => 26,  'name' => 'WordPress',   'icon' => 'fab fa-wordpress',   'desc' => 'Most popular CMS & blogging platform',           'color' => '#21759b'],
                ['id' => 420, 'name' => 'Joomla',      'icon' => 'fas fa-layer-group',  'desc' => 'Flexible & powerful CMS',                        'color' => '#f0851b'],
                ['id' => 31,  'name' => 'Drupal',       'icon' => 'fab fa-drupal',       'desc' => 'Enterprise-level content management',            'color' => '#0077c0'],
                ['id' => 454, 'name' => 'TYPO3',        'icon' => 'fas fa-cube',         'desc' => 'Professional open source CMS',                   'color' => '#f49700'],
            ],
            'E-Commerce' => [
                ['id' => 432, 'name' => 'WooCommerce',  'icon' => 'fas fa-shopping-cart','desc' => 'WordPress-based e-commerce solution',            'color' => '#7f54b3'],
                ['id' => 33,  'name' => 'PrestaShop',   'icon' => 'fas fa-store',        'desc' => 'Feature-rich online store platform',             'color' => '#df0067'],
                ['id' => 35,  'name' => 'OpenCart',     'icon' => 'fas fa-shopping-bag', 'desc' => 'Simple & lightweight shopping cart',             'color' => '#23a1d1'],
                ['id' => 218, 'name' => 'Magento',      'icon' => 'fas fa-gem',          'desc' => 'Enterprise e-commerce platform',                 'color' => '#ee672f'],
            ],
            'Forum' => [
                ['id' => 56,  'name' => 'phpBB',        'icon' => 'fas fa-comments',     'desc' => 'Free & open source forum software',              'color' => '#1a67a3'],
                ['id' => 57,  'name' => 'MyBB',         'icon' => 'fas fa-comment-dots', 'desc' => 'Free discussion board software',                 'color' => '#00aaef'],
                ['id' => 265, 'name' => 'SMF',          'icon' => 'fas fa-users',        'desc' => 'Simple Machines Forum',                         'color' => '#1b75bb'],
                ['id' => 473, 'name' => 'XenForo',      'icon' => 'fas fa-user-friends', 'desc' => 'Modern community forum software',                'color' => '#2e7cbf'],
            ],
            'Blog' => [
                ['id' => 38,  'name' => 'Ghost',        'icon' => 'fab fa-ghost',        'desc' => 'Modern blogging & publishing platform',         'color' => '#15171a'],
                ['id' => 358, 'name' => 'Dotclear',     'icon' => 'fas fa-pen-nib',      'desc' => 'Open source blog engine',                       'color' => '#e07b39'],
            ],
            'Wiki' => [
                ['id' => 75,  'name' => 'MediaWiki',    'icon' => 'fas fa-book-open',    'desc' => 'The wiki engine behind Wikipedia',               'color' => '#3d6e96'],
                ['id' => 76,  'name' => 'DokuWiki',     'icon' => 'fas fa-file-alt',     'desc' => 'Simple, no-database wiki',                      'color' => '#87b3c4'],
            ],
            'Others' => [
                ['id' => 54,  'name' => 'Moodle',       'icon' => 'fas fa-graduation-cap','desc' => 'Learning management system (LMS)',             'color' => '#f98012'],
                ['id' => 215, 'name' => 'SuiteCRM',     'icon' => 'fas fa-briefcase',    'desc' => 'Open source CRM solution',                      'color' => '#1a2030'],
                ['id' => 231, 'name' => 'Roundcube',    'icon' => 'fas fa-envelope',     'desc' => 'Web-based email client',                        'color' => '#37beff'],
                ['id' => 26,  'name' => 'WP Multisite', 'icon' => 'fab fa-wordpress-simple','desc' => 'WordPress network / multi-site install',    'color' => '#21759b'],
            ],
        ];
    }
}
