<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Services\CPanelService;
use App\Models\UserDomain;

class PhpVersionController extends Controller
{
    protected CPanelService $cpanel;

    public function __construct()
    {
        $this->cpanel = new CPanelService(
            env('CPANEL_DOMAIN'),
            env('CPANEL_USERNAME'),
            env('CPANEL_API_TOKEN')
        );
    }

    private function userDomains(): array
    {
        $user    = Auth::user();
        $domains = $user->primary_domain ? [$user->primary_domain] : [];
        $addon   = UserDomain::where('user_id', $user->id)->pluck('domain')->toArray();
        return array_unique(array_merge($domains, $addon));
    }

    private function userOwnsDomain(string $domain): bool
    {
        return in_array($domain, $this->userDomains(), true);
    }

    // Common PHP versions shown when API fails
    private const FALLBACK_VERSIONS = [
        'ea-php83', 'ea-php82', 'ea-php81',
        'ea-php80', 'ea-php74', 'ea-php73', 'ea-php72',
    ];

    /** Convert ea-php81 → "8.1",  php8.1 → "8.1",  8.1 → "8.1" */
    public static function versionLabel(string $ver): string
    {
        if ($ver === 'inherit') return 'System Default';
        // ea-php83 → 83 → 8.3
        if (preg_match('/php(\d)(\d+)$/i', $ver, $m)) {
            return $m[1] . '.' . $m[2];
        }
        // already like "8.1" or "8.1.0"
        if (preg_match('/^(\d+\.\d+)/', $ver, $m)) {
            return $m[1];
        }
        return $ver;
    }

    // Normalize whatever cPanel returns into a flat array of version strings
    private function normalizeVersions(array $raw): array
    {
        $out = [];
        foreach ($raw as $item) {
            if (is_string($item) && $item !== '') {
                $out[] = $item;
            } elseif (is_array($item)) {
                $v = $item['version'] ?? ($item['php_version'] ?? ($item['php'] ?? ($item['ver'] ?? null)));
                if ($v && is_string($v)) $out[] = $v;
            }
        }
        return array_values(array_unique($out));
    }

    // ── Main page ──────────────────────────────────────────────

    public function index()
    {
        $userDomains  = $this->userDomains();
        $apiFailed    = false;

        // ── Installed PHP versions ─────────────────────────────
        $installed = [];
        try {
            $raw       = $this->cpanel->getPhpInstalledVersions();
            $installed = $this->normalizeVersions($raw);
        } catch (\Throwable $e) {
            \Log::warning('PHP versions fetch failed: ' . $e->getMessage());
        }

        if (empty($installed)) {
            $apiFailed = true;
            $installed = self::FALLBACK_VERSIONS;
        }

        // ── Current PHP version per vhost ──────────────────────
        $vhostVersions = [];
        try {
            $all = $this->cpanel->getAllDomainPhpVersions();
            foreach ($all as $item) {
                if (!is_array($item)) continue;
                $vhost = $item['vhost'] ?? ($item['domain'] ?? ($item['servername'] ?? null));
                $ver   = $item['php_version'] ?? ($item['version'] ?? ($item['phpversion'] ?? 'inherit'));
                if ($vhost && is_string($vhost)) {
                    $vhostVersions[strtolower($vhost)] = $ver;
                }
            }
        } catch (\Throwable $e) {
            \Log::warning('PHP vhost versions fetch failed: ' . $e->getMessage());
        }

        // ── Build domain → version map (user's domains only) ───
        $domains = [];
        foreach ($userDomains as $d) {
            $domains[$d] = $vhostVersions[strtolower($d)] ?? 'inherit';
        }

        return view('dashboard.php_version', compact('domains', 'installed', 'apiFailed'));
    }

    // ── AJAX: debug raw API response (admin/dev use) ───────────
    public function debugApi()
    {
        try {
            $versions = $this->cpanel->getPhpInstalledVersions();
            $vhosts   = $this->cpanel->getAllDomainPhpVersions();
            return response()->json([
                'installed_raw' => $versions,
                'vhosts_raw'    => $vhosts,
                'normalized'    => $this->normalizeVersions($versions),
            ]);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()]);
        }
    }

    // ── AJAX: set PHP version ──────────────────────────────────

    public function setVersion(Request $request)
    {
        $request->validate([
            'domain'  => 'required|string',
            'version' => 'required|string',
        ]);

        if (!$this->userOwnsDomain($request->domain))
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);

        $res = $this->cpanel->setDomainPhpVersion($request->domain, $request->version);

        // UAPI success check
        if (isset($res['status']) && (int)$res['status'] === 0) {
            $err = $res['errors'][0] ?? 'Failed to update PHP version';
            return response()->json(['success' => false, 'message' => $err]);
        }

        return response()->json(['success' => true, 'version' => $request->version]);
    }
}
