<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Services\CPanelService;
use App\Models\UserDomain;

class DomainManagerController extends Controller
{
    protected CPanelService $cpanel;

    public function __construct()
    {
        $this->cpanel = new CPanelService();
    }

    // ── helpers ────────────────────────────────────────────────

    /** All domains this user can manage */
    private function userDomains(): array
    {
        $user = Auth::user();
        $domains = [];

        if ($user->primary_domain) {
            $domains[] = $user->primary_domain;
        }

        $addon = UserDomain::where('user_id', $user->id)
            ->pluck('domain')
            ->toArray();

        return array_unique(array_merge($domains, $addon));
    }

    private function userOwnsDomain(string $domain): bool
    {
        return in_array($domain, $this->userDomains(), true);
    }

    private function apiError(array $res): ?string
    {
        // UAPI format: { status: 0, errors: [...] }
        if (isset($res['status']) && (int) $res['status'] === 0) {
            $errs = $res['errors'] ?? [];
            return is_array($errs) ? ($errs[0] ?? 'API error') : (string) $errs;
        }
        // API2 format: { cpanelresult: { data: [{ result: 0, reason: '...' }] } }
        $r2 = $res['cpanelresult']['data'][0] ?? null;
        if ($r2 !== null && isset($r2['result']) && (int) $r2['result'] === 0) {
            return $r2['reason'] ?? ($r2['output'] ?? 'API2 error');
        }
        return null;
    }

    // ── main page ──────────────────────────────────────────────

    public function index()
    {
        $userDomains = $this->userDomains();

        // Get subdomains for each user domain
        $allSubdomains = [];
        try {
            $res = $this->cpanel->listSubdomains();
            $subs = $res['data'] ?? [];
            foreach ($subs as $s) {
                $root = $s['rootdomain'] ?? '';
                if (in_array($root, $userDomains, true)) {
                    $allSubdomains[$root][] = $s;
                }
            }
        } catch (\Throwable) {
        }

        return view('dashboard.domain_manager', compact('userDomains', 'allSubdomains'));
    }

    // ── AJAX: get DNS records ──────────────────────────────────

    public function getDns(string $domain)
    {
        if (!$this->userOwnsDomain($domain))
            return response()->json(['error' => 'Unauthorized'], 403);

        $res = $this->cpanel->getDnsZone($domain);
        $data = $res['data'] ?? [];

        // Normalize records
        $records = array_filter((array) $data, fn($r) => !empty($r['type']));
        $records = array_values($records);

        return response()->json(['records' => $records]);
    }

    // ── AJAX: add DNS record ───────────────────────────────────

    public function addDns(Request $request, string $domain)
    {
        if (!$this->userOwnsDomain($domain))
            return response()->json(['error' => 'Unauthorized'], 403);

        $request->validate([
            'type' => 'required|in:A,AAAA,CNAME,MX,TXT,SRV,CAA',
            'name' => 'required|string',
            'ttl' => 'sometimes|integer',
        ]);

        $record = [
            'type' => $request->type,
            'name' => $request->name,
            'ttl' => $request->ttl ?? 14400,
        ];

        // Type-specific fields
        switch ($request->type) {
            case 'A':
            case 'AAAA':
                $record['address'] = $request->address ?? '';
                break;
            case 'CNAME':
                $record['cname'] = $request->cname ?? '';
                break;
            case 'MX':
                $record['exchange'] = $request->exchange ?? '';
                $record['preference'] = $request->preference ?? 10;
                break;
            case 'TXT':
                $record['txtdata'] = $request->txtdata ?? '';
                break;
            case 'SRV':
                $record['priority'] = $request->priority ?? 10;
                $record['weight'] = $request->weight ?? 0;
                $record['port'] = $request->port ?? 80;
                $record['target'] = $request->target ?? '';
                break;
        }

        $res = $this->cpanel->addDnsRecord($domain, $record);
        if ($err = $this->apiError($res))
            return response()->json(['success' => false, 'message' => $err]);

        return response()->json(['success' => true]);
    }

    // ── AJAX: delete DNS record ────────────────────────────────

    public function deleteDns(Request $request, string $domain)
    {
        if (!$this->userOwnsDomain($domain))
            return response()->json(['error' => 'Unauthorized'], 403);

        $request->validate(['line' => 'required|integer']);
        $res = $this->cpanel->removeDnsRecord($domain, (int) $request->line);
        if ($err = $this->apiError($res))
            return response()->json(['success' => false, 'message' => $err]);

        return response()->json(['success' => true]);
    }

    // ── AJAX: create subdomain ────────────────────────────────

    public function createSubdomain(Request $request)
    {
        $request->validate([
            'subdomain' => 'required|string|regex:/^[a-zA-Z0-9\-]+$/|max:63',
            'root_domain' => 'required|string',
        ]);

        $sub = strtolower(trim($request->subdomain));
        $root = strtolower(trim($request->root_domain));

        if (!$this->userOwnsDomain($root))
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);

        $res = $this->cpanel->createSubdomain($sub, $root);
        if ($err = $this->apiError($res))
            return response()->json(['success' => false, 'message' => $err]);

        // Ensure physical folder exists under domains/<root>/<sub>, like addons
        try {
            $this->cpanel->createFolder("domains/{$root}", $sub);
        } catch (\Throwable $e) {
        }

        // Persist in DB like addon domains (so it doesn't "disappear")
        UserDomain::updateOrCreate(
            ['domain' => $sub . '.' . $root],
            ['user_id' => \Auth::id(), 'subdomain' => $sub]
        );

        return response()->json([
            'success' => true,
            'full' => $sub . '.' . $root,
            'dir' => "domains/{$root}/{$sub}",
        ]);
    }

    // ── AJAX: delete subdomain ────────────────────────────────

    public function deleteSubdomain(Request $request)
    {
        $request->validate(['subdomain' => 'required|string']);

        // Verify ownership: subdomain's root must belong to user
        $parts = explode('.', $request->subdomain, 2);
        $root = $parts[1] ?? '';
        if (!$root || !$this->userOwnsDomain($root))
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);

        $res = $this->cpanel->deleteSubdomain($request->subdomain);
        if ($err = $this->apiError($res))
            return response()->json(['success' => false, 'message' => $err]);

        // Remove from DB if exists
        UserDomain::where('domain', $request->subdomain)->delete();

        return response()->json(['success' => true]);
    }
}
