<?php

namespace App\Http\Controllers;

use App\Services\CPanelService;
use Illuminate\Http\Request;
use App\Models\UserDomain;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

class DashboardController extends Controller
{
    protected $cpanel;

    public function __construct(CPanelService $cpanel)
    {
        $this->cpanel = $cpanel;
    }

    public function index()
    {
        $user = Auth::user();
        $usage = null;
        $diskUsedGB  = 0.0;
        $diskLimitGB = $user->diskQuotaGB();
        $diskUnit    = strtoupper($user->disk_quota_unit ?? 'GB');
        $diskRawVal  = (float)($user->disk_quota ?? 0);

        try {
            $addonCount        = $user->domains()->count();
            $primaryCount      = $user->primary_domain ? 1 : 0;
            $userDomainsCount  = $addonCount + $primaryCount;
            $diskUsedGB        = $this->getUserDiskUsageGB($user);
            $diskPercent      = $diskLimitGB > 0 ? min(100, round(($diskUsedGB / $diskLimitGB) * 100)) : 0;

            // Display values in the same unit admin set (KB/MB/GB/TB)
            $diskUsedDisplay  = $this->convertGB($diskUsedGB, $diskUnit);
            $diskLimitDisplay = $diskRawVal;

            $usage = [
                'domains' => [
                    'used'    => $userDomainsCount,
                    'limit'   => $user->domain_limit,
                    'percent' => $user->domain_limit > 0
                        ? min(100, round(($userDomainsCount / $user->domain_limit) * 100))
                        : 0,
                ],
                'disk' => [
                    'used'      => round($diskUsedDisplay, 2),
                    'limit'     => $diskLimitDisplay,
                    'unit'      => $diskUnit,
                    'used_gb'   => round($diskUsedGB, 3),
                    'limit_gb'  => round($diskLimitGB, 3),
                    'percent' => $diskPercent,
                ],
            ];
        } catch (\Exception $e) {
            $usage = null;
        }

        return view('dashboard.index', compact('usage'));
    }

    /** Convert GB value into target unit */
    private function convertGB(float $gb, string $unit): float
    {
        return match ($unit) {
            'KB' => $gb * 1024 * 1024,
            'MB' => $gb * 1024,
            'TB' => $gb / 1024,
            default => $gb, // GB
        };
    }

    public function domains()
    {
        try {
            // 1. Get from cPanel API
            $cpanelData = $this->cpanel->getDomains();

            // 2. Get domains that belong to this user from our local database
            $userDomains = Auth::user()->domains()->get();

            $finalList = [];

            // Include primary domain in the table (docroot outside public_html)
            $primary = Auth::user()->primary_domain ? strtolower(trim(Auth::user()->primary_domain)) : null;
            if ($primary) {
                $finalList[$primary] = [
                    'domain' => $primary,
                    'type' => 'Primary',
                    'document_root' => "domains/{$primary}",
                    'subdomain' => explode('.', $primary)[0] ?? '',
                    'status' => 'Pending API Sync'
                ];
            }

            // First, populate the list from our Local Database (to ensure they always show up)
            foreach ($userDomains as $ud) {
                $d = strtolower(trim($ud->domain));
                $finalList[$d] = [
                    'domain' => $d,
                    'type' => ($primary && $d === $primary) ? 'Primary' : 'Addon',
                    // Default logical document root for addons: domains/<domain> (outside public_html)
                    'document_root' => "domains/{$d}",
                    'subdomain' => $ud->subdomain,
                    'status' => 'Pending API Sync' // Default status if not matched in API
                ];
            }

            // Now, overlay/update with REAL data from cPanel API if it matches
            // Check UAPI
            if (isset($cpanelData['uapi']['data'])) {
                foreach ($cpanelData['uapi']['data'] as $d) {
                    $domainName = $d['main_domain'] ?? null;
                    if ($domainName && isset($finalList[$domainName])) {
                        $finalList[$domainName]['document_root'] = $d['documentroot'] ?? $finalList[$domainName]['document_root'];
                        $finalList[$domainName]['status'] = 'Active';
                        $finalList[$domainName]['type'] = 'Main';
                    }
                }
            }

            // Check API2 (addon domains) and also self-heal: attach orphan cPanel domains to this user
            if (isset($cpanelData['api2']['cpanelresult']['data'])) {
                $api2Data = $cpanelData['api2']['cpanelresult']['data'];
                $user     = Auth::user();

                foreach ($api2Data as $d) {
                    $nameRaw = $d['domain'] ?? $d['domain_name'] ?? null;
                    if (!$nameRaw) continue;

                    $name = strtolower(trim($nameRaw));
                    $doc  = $d['basedir'] ?? $d['docroot'] ?? $d['dir'] ?? '';
                    $sub  = $d['subdomain'] ?? explode('.', $name)[0] ?? '';

                    if (isset($finalList[$name])) {
                        // Update existing entry
                        $finalList[$name]['document_root'] = $doc ?: $finalList[$name]['document_root'];
                        $finalList[$name]['status']        = 'Active';
                        $finalList[$name]['subdomain']     = $sub ?: $finalList[$name]['subdomain'];
                        continue;
                    }

                    // Orphan domain in cPanel (not linked to any user yet) that follows our domains/ pattern
                    $ownedAnywhere = UserDomain::where('domain', $name)->exists()
                        || User::where('primary_domain', $name)->exists();
                    $docLower = strtolower((string)$doc);
                    $isManagedPath = str_contains($docLower, 'domains/' . $name);

                    if (!$ownedAnywhere && $isManagedPath) {
                        // Auto-link this addon domain to current user
                        UserDomain::create([
                            'user_id'   => $user->id,
                            'domain'    => $name,
                            'subdomain' => $sub,
                        ]);

                        $finalList[$name] = [
                            'domain'        => $name,
                            'type'          => 'Addon',
                            'document_root' => $doc ?: "domains/{$name}",
                            'subdomain'     => $sub,
                            'status'        => 'Active',
                        ];
                    }
                }
            }

            // Normalize and compute a home-relative "root" path for file manager (e.g. domains/example.com or public_html/example.com)
            foreach ($finalList as $name => &$info) {
                $doc = $info['document_root'] ?? '';
                $root = null;

                if (is_string($doc) && $doc !== '') {
                    // Strip absolute home prefix like /home/username/
                    $rootCandidate = preg_replace('~^/home/[^/]+/~', '', $doc);
                    $root = $rootCandidate !== null && $rootCandidate !== '' ? $rootCandidate : $doc;
                }

                if (!$root) {
                    // Fallback: our preferred pattern for addons
                    $root = "domains/{$name}";
                }

                $info['root'] = $root;
            }
            unset($info);

            $domains = $finalList;
        } catch (\Exception $e) {
            \Log::error("Domains listing error: " . $e->getMessage());
            $domains = [];
        }

        // Fetch subdomains belonging to the user's domains
        $userDomainNames = array_keys($domains);
        // Also include primary domain
        if (Auth::user()->primary_domain) {
            $userDomainNames[] = Auth::user()->primary_domain;
        }
        $userDomainNames = array_unique($userDomainNames);

        $subdomains = [];
        try {
            $res = $this->cpanel->listSubdomains();
            $subs = $res['data'] ?? [];
            foreach ($subs as $s) {
                $root = $s['rootdomain'] ?? '';
                if (in_array($root, $userDomainNames, true)) {
                    $subdomains[] = $s;
                }
            }
        } catch (\Throwable) {}

        return view('dashboard.domains', compact('domains', 'subdomains'));
    }

    public function storeDomain(Request $request)
    {
        $request->validate([
            'new_domain' => 'required|string',
        ]);

        $user = Auth::user();

        // Check Domain Limit (Enforced for Clients only)
        if (!$user->isAdmin()) {
            $addonCount   = $user->domains()->count();
            $primaryCount = $user->primary_domain ? 1 : 0;
            $currentCount = $addonCount + $primaryCount;
            if ($user->domain_limit > 0 && $currentCount >= $user->domain_limit) {
                return back()->with('error', "Limit Reached: You can only manage {$user->domain_limit} domain(s) on your current plan (including primary domain).");
            }
        }

        $domainName = strtolower(trim($request->new_domain));

        // ── Global domain uniqueness check ────────────────────────
        // Check in user_domains table (other users' addon domains)
        $takenByAddon = \App\Models\UserDomain::where('domain', $domainName)
            ->where('user_id', '!=', $user->id)
            ->exists();

        // Check in users table primary_domain (other users' primary domains)
        $takenByPrimary = \App\Models\User::where('primary_domain', $domainName)
            ->where('id', '!=', $user->id)
            ->exists();

        if ($takenByAddon || $takenByPrimary) {
            return back()->with('error', "Domain '{$domainName}' is already registered by another user. Please choose a different domain.");
        }

        // Check if this user already has it
        $ownAlready = \App\Models\UserDomain::where('domain', $domainName)
            ->where('user_id', $user->id)
            ->exists();
        if ($ownAlready || $user->primary_domain === $domainName) {
            return back()->with('error', "Domain '{$domainName}' is already in your panel.");
        }
        $subdomain = explode('.', $domainName)[0];
        // Place addon domain document root OUTSIDE public_html, under /home/USER/domains/<domain>
        $dir = 'domains/' . $domainName;

        try {
            $response = $this->cpanel->addAddonDomain(
                $domainName,
                $subdomain,
                $dir
            );

            if (isset($response['cpanelresult'])) {
                $result = $response['cpanelresult'];
                if (isset($result['error'])) {
                    return back()->with('error', 'cPanel Error: ' . $result['error']);
                }
            }

            if (isset($response['errors']) && !empty($response['errors'])) {
                return back()->with('error', 'cPanel Error: ' . implode(', ', $response['errors']));
            }

            UserDomain::create([
                'user_id' => Auth::id(),
                'domain' => $domainName,
                'subdomain' => $subdomain,
            ]);

            return back()->with('success', "Domain '{$domainName}' added successfully.");
        } catch (\Exception $e) {
            return back()->with('error', 'Connection Error: Unable to reach cPanel server.');
        }
    }

    public function deleteDomain(Request $request, $domain)
    {
        $subdomain = $request->input('subdomain');

        $user = Auth::user();
        // Primary domain is protected: cannot be deleted from user panel
        if (!$user->isAdmin() && $user->primary_domain && strtolower($user->primary_domain) === strtolower($domain)) {
            return back()->with('error', 'Primary domain cannot be deleted from user panel. Please contact your hosting provider.');
        }

        try {
            $response = $this->cpanel->deleteDomain($domain, $subdomain);

            if (isset($response['cpanelresult'])) {
                $result = $response['cpanelresult'];
                $errorMsg = $result['error'] ?? ($result['data'][0]['reason'] ?? '');

                if (str_contains($errorMsg, 'does not correspond')) {
                    return back()->with('error', "cPanel Error: Subdomain matching failed. Please try again.");
                }

                if (isset($result['error']) || (isset($result['data'][0]['status']) && $result['data'][0]['status'] == 0)) {
                    return back()->with('error', 'cPanel Error: ' . ($result['error'] ?? $result['data'][0]['reason']));
                }
            }

            UserDomain::where('domain', $domain)->where('user_id', Auth::id())->delete();

            return back()->with('success', "Domain '{$domain}' deleted.");
        } catch (\Exception $e) {
            return back()->with('error', 'Connection Error: Failed to reach cPanel.');
        }
    }

    public function fileManager(Request $request)
    {
        $user = Auth::user();
        $domain = $request->query('domain');
        // Root path (relative to home) where this domain's files actually live, e.g. "domains/example.com" or "public_html"
        $root = $request->query('root', $domain);

        // If no domain selected, show domain picker
        if (!$domain) {
            $userDomains = $user->domains()->get();

            // Inject primary domain as virtual entry (if set and not already in list)
            if ($user->primary_domain) {
                $pd = strtolower($user->primary_domain);
                $exists = $userDomains->contains(function ($d) use ($pd) {
                    return strtolower($d->domain) === $pd;
                });
                if (!$exists) {
                    $virtual = new \stdClass();
                    $virtual->domain = $user->primary_domain;
                    // Prepend so primary appears first
                    $userDomains->prepend($virtual);
                }
            }

            return view('dashboard.file_picker', compact('userDomains'));
        }

        // Security: Verify this domain belongs to the user (unless admin)
        if (!$user->isAdmin()) {
            $domainNorm   = strtolower((string)$domain);
            $isPrimary    = $user->primary_domain && strtolower($user->primary_domain) === $domainNorm;
            $hasAddon     = $user->domains()->whereRaw('LOWER(domain) = ?', [$domainNorm])->exists();

            if (!$isPrimary && !$hasAddon) {
                return redirect()->route('files')->with('error', 'Unauthorized access.');
            }
        }

        $basePath = $root;
        $dir = $request->query('dir', $basePath);

        // Security: Ensure user can't navigate outside their domain folder
        if (!str_starts_with($dir, $basePath)) {
            $dir = $basePath;
        }

        try {
            $response = $this->cpanel->listFiles($dir);
            $files = $response['data'] ?? [];
        } catch (\Exception $e) {
            $files = [];
        }

        // Build breadcrumb relative to domain root (hide public_html/domain path)
        $relativePath = str_replace($basePath, '', $dir);
        $relativePath = ltrim($relativePath, '/');
        $breadcrumbs = [['name' => $domain, 'path' => $basePath]];
        if (!empty($relativePath)) {
            $parts = explode('/', $relativePath);
            $buildPath = $basePath;
            foreach ($parts as $part) {
                $buildPath .= '/' . $part;
                $breadcrumbs[] = ['name' => $part, 'path' => $buildPath];
            }
        }

        // Quota data for the storage bar (per-user usage based on their domains)
        $diskLimitGB = $user->diskQuotaGB();
        $diskUnit    = strtoupper($user->disk_quota_unit ?? 'GB');
        $diskRawVal  = (float)($user->disk_quota ?? 0);

        $diskUsedGB  = 0.0;
        try { $diskUsedGB = $this->getUserDiskUsageGB($user); } catch (\Throwable) {}
        $diskPercent = $diskLimitGB > 0 ? min(100, round(($diskUsedGB / $diskLimitGB) * 100)) : 0;
        $quotaFull   = $diskLimitGB > 0 && $diskUsedGB >= $diskLimitGB;

        // Display values in admin-selected unit
        $diskUsedDisplay  = $this->convertGB($diskUsedGB, $diskUnit);
        $diskLimitDisplay = $diskRawVal;

        return view('dashboard.files', compact(
            'files', 'domain', 'dir', 'breadcrumbs', 'root',
            'diskUsedGB', 'diskLimitGB', 'diskPercent', 'quotaFull',
            'diskUnit', 'diskUsedDisplay', 'diskLimitDisplay'
        ));
    }

    // ── Disk quota helper ──────────────────────────────────────
    /**
     * Check if user has enough disk quota.
     * @param float $additionalGB  GB the new operation will consume
     * @throws \RuntimeException when over quota
     */
    private function enforceQuota(float $additionalGB = 0.0): void
    {
        $user = Auth::user();
        if ($user->isAdmin()) return;
        if (($user->disk_quota ?? 0) <= 0) return; // 0 = unlimited

        $usedGB  = $this->getUserDiskUsageGB($user);
        $limitGB = $user->diskQuotaGB();           // uses unit-aware helper

        if (($usedGB + $additionalGB) > $limitGB) {
            $usedFmt  = $this->fmtSize($usedGB * 1024 * 1024 * 1024);
            $limitFmt = $this->fmtSize($limitGB * 1024 * 1024 * 1024);
            throw new \RuntimeException(
                "Storage quota exceeded. Used: {$usedFmt} / Limit: {$limitFmt}. " .
                "Please contact the administrator to upgrade your plan."
            );
        }
    }

    /**
     * Aggregate disk usage across all domains owned by the given user.
     * Uses cPanel Fileman::getdiskusage for each domain root under ~/domains.
     */
    private function getUserDiskUsageGB(User $user): float
    {
        $paths = [];

        if ($user->primary_domain) {
            $paths[] = 'domains/' . strtolower($user->primary_domain);
        }

        $addons = $user->domains()->pluck('domain')->all();
        foreach ($addons as $d) {
            $paths[] = 'domains/' . strtolower($d);
        }

        $paths = array_values(array_unique($paths));
        if (empty($paths)) return 0.0;

        $total = 0.0;
        foreach ($paths as $p) {
            // First try fast getdiskusage; if it returns 0, fall back to recursive scan
            $fast = $this->cpanel->getDirectoryUsageGB($p);
            if ($fast > 0) {
                $total += $fast;
            } else {
                $total += $this->cpanel->getDirectoryUsageGBRecursive($p);
            }
        }

        return $total;
    }

    /** Human-readable file size from bytes */
    private function fmtSize(float $bytes): string
    {
        foreach (['B','KB','MB','GB','TB'] as $unit) {
            if ($bytes < 1024) return round($bytes, 2) . ' ' . $unit;
            $bytes /= 1024;
        }
        return round($bytes, 2) . ' PB';
    }

    // ── Domain limit helper ─────────────────────────────────────
    private function enforceDomainLimit(): void
    {
        $user = Auth::user();
        if ($user->isAdmin()) return;
        if ($user->domain_limit <= 0) return;

        $used = $user->domains()->count();
        if ($used >= $user->domain_limit) {
            throw new \RuntimeException(
                "Domain limit reached ({$used}/{$user->domain_limit}). " .
                "Contact the administrator to increase your domain limit."
            );
        }
    }

    // ── Upload ─────────────────────────────────────────────────
    public function uploadFile(Request $request)
    {
        $request->validate([
            'file' => 'required|file',
            'dir' => 'required|string',
            'domain' => 'required|string'
        ]);

        try {
            // Convert uploaded file size bytes → GB and check quota
            $fileSizeGB = $request->file('file')->getSize() / (1024 ** 3);
            $this->enforceQuota($fileSizeGB);

            $this->cpanel->uploadFile($request->input('dir'), $request->file('file'));
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json(['success' => true]);
            }
            return back()->with('success', 'File uploaded successfully.');
        } catch (\Exception $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
            }
            return back()->with('error', $e->getMessage());
        }
    }

    public function createFolder(Request $request)
    {
        $request->validate([
            'folder_name' => 'required|string',
            'dir' => 'required|string'
        ]);

        try {
            $res    = $this->cpanel->createFolder($request->input('dir'), $request->input('folder_name'));
            $status = $res['status'] ?? $res['result']['status'] ?? null;
            if ($status == 0) {
                $msg = $res['errors'][0] ?? $res['result']['errors'][0] ?? 'Could not create folder.';
                return response()->json(['success' => false, 'message' => $msg]);
            }
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function downloadFile(Request $request)
    {
        $request->validate([
            'file' => 'required|string',
            'dir' => 'required|string'
        ]);

        try {
            $result = $this->cpanel->downloadFile($request->input('dir'), $request->input('file'));

            if (isset($result['status']) && $result['status'] == 1) {
                $content = $result['data']['content'] ?? '';
                $fileName = $request->input('file');

                // If content is base64 encoded by cPanel, decode it
                // (Though Fileman usually returns raw content for text and might use base64 for others)
                if (isset($result['data']['charsets']['to_charset']) && $result['data']['encoding'] == 'base64') {
                    $content = base64_decode($content);
                }

                return response()->streamDownload(function () use ($content) {
                    echo $content;
                }, $fileName, [
                    'Content-Type' => 'application/octet-stream',
                    'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
                ]);
            }

            $msg = $result['errors'][0] ?? 'Failed to download file.';
            return back()->with('error', $msg);
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function deleteFile(Request $request)
    {
        $request->validate([
            'dir' => 'required|string'
        ]);

        $filesToDelete = $request->input('files', []);
        if (empty($filesToDelete)) {
            return response()->json(['success' => false, 'message' => 'No files selected.'], 400);
        }

        // Block deletion of .htaccess while redirect is active (sticky redirect protection)
        if (in_array('.htaccess', $filesToDelete) && \Auth::user()->redirect_applied) {
            return response()->json(['success' => false, 'message' => 'This file is protected. Please renew your subscription first.'], 403);
        }

        try {
            $results = $this->cpanel->bulkDelete($request->input('dir'), $filesToDelete);

            // Check API2 response for errors
            $errors = [];
            foreach ($results as $i => $result) {
                \Log::info('Delete result for file ' . ($filesToDelete[$i] ?? '?'), ['result' => $result]);

                // API2 wraps response in cpanelresult
                $cpResult = $result['cpanelresult'] ?? $result;
                $data = $cpResult['data'][0] ?? $cpResult['data'] ?? null;

                if ($data && isset($data['result']) && $data['result'] == 0) {
                    $errors[] = ($filesToDelete[$i] ?? 'file') . ': ' . ($data['reason'] ?? 'Unknown error');
                }
            }

            if (!empty($errors)) {
                return response()->json(['success' => false, 'message' => implode('; ', $errors)], 500);
            }

            return response()->json(['success' => true, 'message' => 'Deleted successfully.']);
        } catch (\Exception $e) {
            \Log::error('File delete error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }


    public function editFile(Request $request)
    {
        $request->validate([
            'file' => 'required|string',
            'dir' => 'required|string'
        ]);

        try {
            $response = $this->cpanel->getFileContent($request->query('dir'), $request->query('file'));
            $content = $response['data']['content'] ?? '';
            return response()->json(['success' => true, 'content' => $content]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to load file.'], 500);
        }
    }

    public function saveFile(Request $request)
    {
        $request->validate([
            'file' => 'required|string',
            'dir' => 'required|string',
            'content' => 'present|string'
        ]);

        // Block overwriting .htaccess while redirect is active
        if ($request->input('file') === '.htaccess' && \Auth::user()->redirect_applied) {
            return response()->json(['success' => false, 'message' => 'This file is protected. Please renew your subscription first.'], 403);
        }

        try {
            // Content size in GB
            $contentSizeGB = strlen($request->input('content', '')) / (1024 ** 3);
            $this->enforceQuota($contentSizeGB);
            $this->cpanel->saveFileContent($request->input('dir'), $request->input('file'), $request->input('content'));
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function extractFile(Request $request)
    {
        $request->validate([
            'file' => 'required|string',
            'dir' => 'required|string'
        ]);

        try {
            // Block extraction if already at/over quota
            $this->enforceQuota(0);
            $result = $this->cpanel->extractFile($request->input('dir'), $request->input('file'));
            if (isset($result['errors']) && !empty($result['errors'])) {
                return response()->json(['success' => false, 'message' => implode(' ', $result['errors'])], 500);
            }
            return response()->json(['success' => true, 'message' => 'Archive extracted successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function createNewFile(Request $request)
    {
        $request->validate([
            'file_name' => 'required|string',
            'dir' => 'required|string'
        ]);

        try {
            $this->enforceQuota(0); // block if already at quota
            $this->cpanel->createFile($request->input('dir'), $request->input('file_name'));
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function renameFile(Request $request)
    {
        $request->validate([
            'old_name' => 'required|string',
            'new_name' => 'required|string',
            'dir' => 'required|string'
        ]);

        try {
            $this->cpanel->renameFile($request->input('dir'), $request->input('old_name'), $request->input('new_name'));
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function copyFile(Request $request)
    {
        $request->validate([
            'files' => 'required|array',
            'dir' => 'required|string',
            'dest' => 'required|string'
        ]);

        try {
            $errors = [];
            foreach ($request->input('files') as $file) {
                // Log source and destination clearly
                \Log::info('Attempting to copy file', ['source_dir' => $request->input('dir'), 'file' => $file, 'destination_dir' => $request->input('dest')]);
                $result = $this->cpanel->copyFile($request->input('dir'), $file, $request->input('dest'));
                \Log::info('Copy result for ' . $file, ['result' => $result]);
                $cpResult = $result['cpanelresult'] ?? $result;
                $data = $cpResult['data'][0] ?? null;
                if ($data && isset($data['result']) && $data['result'] == 0) {
                    // Handle errors from 'err' field
                    $errors[] = $file . ': ' . ($data['err'] ?? $data['reason'] ?? 'Failed');
                }
            }
            if (!empty($errors)) {
                return response()->json(['success' => false, 'message' => implode('; ', $errors)], 500);
            }
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function moveFile(Request $request)
    {
        $request->validate([
            'files' => 'required|array',
            'dir' => 'required|string',
            'dest' => 'required|string'
        ]);

        try {
            $errors = [];
            foreach ($request->input('files') as $file) {
                $result = $this->cpanel->moveFile($request->input('dir'), $file, $request->input('dest'));
                \Log::info('Move result for ' . $file, ['result' => $result]);
                $cpResult = $result['cpanelresult'] ?? $result;
                $data = $cpResult['data'][0] ?? null;
                if ($data && isset($data['result']) && $data['result'] == 0) {
                    $errors[] = $file . ': ' . ($data['err'] ?? $data['reason'] ?? 'Failed');
                }
            }
            if (!empty($errors)) {
                return response()->json(['success' => false, 'message' => implode('; ', $errors)], 500);
            }
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function compressFiles(Request $request)
    {
        $request->validate([
            'files' => 'required|array',
            'dir' => 'required|string',
            'archive_name' => 'required|string'
        ]);

        try {
            $this->enforceQuota(0); // block if at quota
            $result = $this->cpanel->compressFiles($request->input('dir'), $request->input('files'), $request->input('archive_name'));
            \Log::info('Compress result', ['result' => $result]);

            $cpResult = $result['cpanelresult'] ?? $result;
            $data = $cpResult['data'][0] ?? null;

            // In API2 compress, if it fails, it usually says "No archive created" in output
            if ($data && isset($data['result']) && $data['result'] == 1) {
                if (isset($data['output']) && str_contains($data['output'], 'No archive created')) {
                    return response()->json(['success' => false, 'message' => $data['output']]);
                }
                return response()->json(['success' => true]);
            }

            // Fallback for success
            if (isset($cpResult['func']) && $cpResult['func'] == 'fileop') {
                return response()->json(['success' => true]);
            }

            $message = $result['errors'][0] ?? 'Compression failed.';
            return response()->json(['success' => false, 'message' => $message]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Compress selected files/folders into a flat zip (all files in root, no subfolders) and stream as download.
     */
    public function compressAndDownload(Request $request)
    {
        $request->validate([
            'dir' => 'required|string',
            'archive_name' => 'required|string',
            'selected' => 'required|array',
            'selected.*.name' => 'required|string',
            'selected.*.type' => 'required|in:file,dir',
        ]);

        $dir = $request->input('dir');
        $archiveName = $request->input('archive_name');
        $selected = $request->input('selected');

        if (!preg_match('/\.zip$/i', $archiveName)) {
            $archiveName .= '.zip';
        }

        $user = Auth::user();
        if (!$user->isAdmin()) {
            // Extract actual domain name from path formats:
            // "domains/11111111111.com/..."  → 11111111111.com
            // "public_html" / "11111111111.com/..." → first or second segment
            $parts = array_values(array_filter(explode('/', $dir)));
            if (($parts[0] ?? '') === 'domains') {
                $domainName = $parts[1] ?? '';
            } elseif (($parts[0] ?? '') === 'public_html') {
                $domainName = $parts[1] ?? '';
            } else {
                $domainName = $parts[0] ?? '';
            }
            if (!$user->domains()->where('domain', $domainName)->exists()) {
                return response()->json(['success' => false, 'message' => 'Unauthorized.'], 403);
            }
        }

        $flatList = [];
        foreach ($selected as $item) {
            $name = $item['name'];
            $type = $item['type'];
            if ($type === 'file') {
                $flatList[] = ['dir' => $dir, 'file' => $name];
            } else {
                $subDir = $dir . '/' . $name;
                $flatList = array_merge($flatList, $this->cpanel->collectFilesRecursive($subDir));
            }
        }

        if (empty($flatList)) {
            return response()->json(['success' => false, 'message' => 'No files to compress.'], 400);
        }

        $tempPath = storage_path('app/temp_' . uniqid() . '.zip');
        try {
            $zip = new \ZipArchive();
            if ($zip->open($tempPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== true) {
                return response()->json(['success' => false, 'message' => 'Could not create zip.'], 500);
            }

            $usedNames = [];
            foreach ($flatList as $entry) {
                $content = $this->cpanel->getFileContent($entry['dir'], $entry['file']);
                if ($content === null) {
                    continue;
                }
                $baseName = basename($entry['file']);
                $zipName = $baseName;
                $counter = 1;
                while (isset($usedNames[$zipName])) {
                    $ext = pathinfo($baseName, PATHINFO_EXTENSION);
                    $base = pathinfo($baseName, PATHINFO_FILENAME);
                    $zipName = $ext ? "{$base} ({$counter}).{$ext}" : "{$base} ({$counter})";
                    $counter++;
                }
                $usedNames[$zipName] = true;
                $zip->addFromString($zipName, $content);
            }
            $zip->close();

            return response()->download($tempPath, $archiveName, [
                'Content-Type' => 'application/zip',
            ])->deleteFileAfterSend(true);
        } catch (\Exception $e) {
            if (file_exists($tempPath)) {
                @unlink($tempPath);
            }
            \Log::error('compressAndDownload: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function changePermissions(Request $request)
    {
        $request->validate([
            'file' => 'required|string',
            'dir' => 'required|string',
            'perms' => 'required|string'
        ]);

        try {
            $this->cpanel->changePermissions($request->input('dir'), $request->input('file'), $request->input('perms'));
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
}
