<?php

namespace App\Http\Controllers;

use App\Models\AdminSetting;
use App\Models\Notice;
use App\Models\SupportMessage;
use App\Models\UserDomain;
use App\Models\User;
use App\Services\CPanelService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Artisan;

class AdminController extends Controller
{
    // ── cPanel domain provisioning helpers ─────────────────────
    private function cpanelError(array $res): ?string
    {
        // UAPI format: { status: 0, errors: [...] }
        if (isset($res['status']) && (int) $res['status'] === 0) {
            $errs = $res['errors'] ?? [];
            return is_array($errs) ? ($errs[0] ?? 'API error') : (string) $errs;
        }

        // API2 format: { cpanelresult: { error: '...', data: [...] } }
        $r = $res['cpanelresult'] ?? null;
        if (is_array($r)) {
            if (!empty($r['error']))
                return (string) $r['error'];
            $d0 = $r['data'][0] ?? null;
            if (is_array($d0) && isset($d0['result']) && (int) $d0['result'] === 0) {
                return (string) ($d0['reason'] ?? $d0['error'] ?? 'API2 error');
            }
        }

        // Generic errors
        if (!empty($res['errors'][0]))
            return (string) $res['errors'][0];
        return null;
    }

    private function buildSubdomain(string $domain, int $attempt = 0): string
    {
        $base = explode('.', $domain)[0] ?? 'd';
        $base = strtolower(preg_replace('/[^a-z0-9\\-]/i', '', $base) ?: 'd');
        if (preg_match('/^[0-9]/', $base))
            $base = 'd' . $base;

        // Keep short to avoid limits and allow suffix
        $base = substr($base, 0, 32);
        if ($attempt <= 0)
            return $base;

        $suffix = (string) random_int(100, 9999);
        $trimTo = max(1, 32 - (1 + strlen($suffix)));
        return substr($base, 0, $trimTo) . '-' . $suffix;
    }

    private function provisionPrimaryDomain(string $domain): void
    {
        $domain = strtolower(trim($domain));
        if ($domain === '')
            return;

        $cpanel = new CPanelService();
        $docRoot = 'domains/' . $domain;

        // Try a few times in case subdomain already exists
        for ($i = 0; $i < 5; $i++) {
            $sub = $this->buildSubdomain($domain, $i);
            $res = $cpanel->addAddonDomain($domain, $sub, $docRoot);
            $err = $this->cpanelError($res);
            if (!$err)
                return;

            $msg = strtolower($err);
            // If domain already exists in cPanel, treat as success
            if (str_contains($msg, 'already exists') || str_contains($msg, 'exists') || str_contains($msg, 'already configured')) {
                return;
            }
            // Retry with different subdomain if conflict mentioned
            if (str_contains($msg, 'subdomain'))
                continue;

            throw new \RuntimeException($err);
        }

        throw new \RuntimeException('Could not provision domain in cPanel (subdomain conflict).');
    }

    // ── Client list ────────────────────────────────────────────
    public function index()
    {
        $clients = User::where('role', 'client')->latest()->get();
        $notices = Notice::latest()->get();
        $unreadChats = SupportMessage::where('sender', 'user')
            ->where('is_read', false)
            ->selectRaw('user_id, COUNT(*) as cnt')
            ->groupBy('user_id')
            ->pluck('cnt', 'user_id');
        $renewUrl = AdminSetting::get('renew_url', '');
        $totalUnreadCount = SupportMessage::where('sender', 'user')->where('is_read', false)->count();

        return view('admin.index', compact('clients', 'notices', 'unreadChats', 'renewUrl', 'totalUnreadCount'));
    }

    // ── Admin settings (renew URL etc.) ───────────────────────
    public function saveSettings(Request $request)
    {
        $request->validate([
            'renew_url' => 'nullable|url|max:500',
            'admin_email' => 'nullable|email|max:255',
        ]);
        AdminSetting::set('renew_url', $request->renew_url ?? '');
        AdminSetting::set('admin_email', $request->admin_email ?? '');
        return response()->json(['success' => true]);
    }

    // ── Admin system tools (artisan helpers) ───────────────────
    public function runTool(Request $request)
    {
        $request->validate([
            'action' => 'required|in:storage_link,cache_clear',
        ]);

        try {
            switch ($request->action) {
                case 'storage_link':
                    Artisan::call('storage:link');
                    break;
                case 'cache_clear':
                    Artisan::call('cache:clear');
                    Artisan::call('config:clear');
                    Artisan::call('route:clear');
                    Artisan::call('view:clear');
                    break;
            }

            return response()->json(['success' => true]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    // ── Unread chat count (AJAX polling) ──────────────────────
    public function unreadCount()
    {
        $total = SupportMessage::where('sender', 'user')->where('is_read', false)->count();
        $perUser = SupportMessage::where('sender', 'user')
            ->where('is_read', false)
            ->selectRaw('user_id, COUNT(*) as cnt')
            ->groupBy('user_id')
            ->pluck('cnt', 'user_id');

        return response()->json(['total' => $total, 'per_user' => $perUser]);
    }

    // ── Create client ──────────────────────────────────────────
    public function storeClient(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8',
            'plan_name' => 'nullable|string|max:100',
            'billing_amount' => 'nullable|numeric|min:0',
            'subscription_expires_at' => 'nullable|date',
            'disk_quota' => 'required|integer|min:1',
            'domain_limit' => 'required|integer|min:1',
            'email_limit' => 'required|integer|min:1',
            'primary_domain' => 'nullable|string|max:255',
        ]);

        $primary = $request->filled('primary_domain')
            ? strtolower(trim($request->primary_domain))
            : null;

        if ($primary) {
            $takenPrimary = User::where('primary_domain', $primary)->exists();
            $takenAddon = UserDomain::where('domain', $primary)->exists();
            if ($takenPrimary || $takenAddon) {
                return back()->with('error', "Domain '{$primary}' is already assigned to another user.");
            }

            try {
                $this->provisionPrimaryDomain($primary);
            } catch (\Throwable $e) {
                return back()->with('error', 'cPanel Error: ' . $e->getMessage());
            }
        }

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => 'client',
            'status' => 'active',
            'plan_name' => $request->plan_name ?? 'Basic',
            'billing_amount' => $request->billing_amount ?? 0,
            'subscription_expires_at' => $request->subscription_expires_at,
            'disk_quota' => $request->disk_quota,
            'domain_limit' => $request->domain_limit,
            'email_limit' => $request->email_limit,
            'primary_domain' => $primary,
        ]);

        return back()->with('success', 'Client account created successfully.');
    }

    // ── Get client data (AJAX for edit modal) ─────────────────
    public function clientData($id)
    {
        $user = User::findOrFail($id);
        return response()->json([
            'name' => $user->name,
            'email' => $user->email,
            'plan_name' => $user->plan_name,
            'billing_amount' => $user->billing_amount,
            'subscription_expires_at' => $user->subscription_expires_at?->format('Y-m-d'),
            'status' => $user->status,
            'primary_domain' => $user->primary_domain,
            'disk_quota' => $user->disk_quota,
            'disk_quota_unit' => $user->disk_quota_unit ?? 'GB',
            'renew_url' => $user->renew_url ?? '',
            'domain_limit' => $user->domain_limit,
            'email_limit' => $user->email_limit,
        ]);
    }

    // ── Edit client (AJAX) ─────────────────────────────────────
    public function editClient(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'plan_name' => 'nullable|string|max:100',
            'billing_amount' => 'nullable|numeric|min:0',
            'subscription_expires_at' => 'nullable|date',
            'status' => 'required|in:active,suspended,expired',
            'disk_quota' => 'required|integer|min:1',
            'domain_limit' => 'required|integer|min:1',
            'email_limit' => 'required|integer|min:1',
            'primary_domain' => 'nullable|string|max:255',
        ]);

        $data = $request->only([
            'name',
            'email',
            'plan_name',
            'billing_amount',
            'subscription_expires_at',
            'status',
            'disk_quota',
            'disk_quota_unit',
            'renew_url',
            'domain_limit',
            'email_limit',
            'primary_domain',
        ]);

        // ── primary_domain global uniqueness check ─────────────
        $oldPrimary = $user->primary_domain ? strtolower(trim($user->primary_domain)) : null;
        if (!empty($data['primary_domain']) && $data['primary_domain'] !== $user->primary_domain) {
            $pd = strtolower(trim($data['primary_domain']));
            $takenPrimary = User::where('primary_domain', $pd)->where('id', '!=', $id)->exists();
            $takenAddon = \App\Models\UserDomain::where('domain', $pd)->where('user_id', '!=', $id)->exists();
            if ($takenPrimary || $takenAddon) {
                return response()->json(['success' => false, 'message' => "Domain '{$pd}' is already assigned to another user."], 422);
            }
            $data['primary_domain'] = $pd;
        }

        // Provision in cPanel if primary domain changed/added
        $newPrimary = !empty($data['primary_domain']) ? (string) $data['primary_domain'] : null;
        if ($newPrimary && $newPrimary !== $oldPrimary) {
            try {
                $this->provisionPrimaryDomain($newPrimary);
            } catch (\Throwable $e) {
                return response()->json(['success' => false, 'message' => 'cPanel Error: ' . $e->getMessage()], 422);
            }
        }

        if ($request->filled('password')) {
            $request->validate(['password' => 'min:8']);
            $data['password'] = Hash::make($request->password);
        }

        $prevStatus = $user->subscriptionStatus();
        $user->update($data);

        $fresh = $user->fresh();
        $newStatus = $fresh->subscriptionStatus();

        // If became suspended/expired → apply redirect + mark applied
        if ($prevStatus === 'active' && in_array($newStatus, ['suspended', 'expired'])) {
            $this->applyDomainRedirects($fresh);
            $fresh->update(['redirect_applied' => true]);
        }
        // If restored to active → remove redirects + reset flag
        elseif ($prevStatus !== 'active' && $newStatus === 'active') {
            $this->removeDomainRedirects($fresh);
            $fresh->update(['redirect_applied' => false]);
        }

        return response()->json(['success' => true, 'message' => 'User updated successfully.']);
    }

    // ── Delete client ──────────────────────────────────────────
    public function deleteClient($id)
    {
        User::findOrFail($id)->delete();
        return back()->with('success', 'Client account deleted.');
    }

    // ── Domain redirect helpers ────────────────────────────────
    private function applyDomainRedirects(User $user): void
    {
        // Priority: user-specific URL → global admin URL → panel's own expired page
        $renewUrl = $user->renew_url
            ?: AdminSetting::get('renew_url', '')
            ?: url('/hosting-expired');

        $cpanel = new CPanelService();

        $domains = UserDomain::where('user_id', $user->id)->get();
        foreach ($domains as $ud) {
            $docRoot = 'domains/' . $ud->domain;
            $cpanel->addDomainRedirect($docRoot, $renewUrl);
        }
        if ($user->primary_domain) {
            $cpanel->addDomainRedirect('domains/' . $user->primary_domain, $renewUrl);
        }
    }

    private function removeDomainRedirects(User $user): void
    {
        $cpanel = new CPanelService();

        $domains = UserDomain::where('user_id', $user->id)->get();
        foreach ($domains as $ud) {
            $cpanel->removeDomainRedirect('domains/' . $ud->domain);
        }
        if ($user->primary_domain) {
            $cpanel->removeDomainRedirect('domains/' . $user->primary_domain);
        }
    }

    // ── Manual toggle domain redirect (AJAX) ──────────────────
    public function toggleRedirect(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $action = $request->input('action', 'apply'); // apply | remove
        if ($action === 'apply') {
            $this->applyDomainRedirects($user);
            $user->update(['redirect_applied' => true]);
        } else {
            $this->removeDomainRedirects($user);
            $user->update(['redirect_applied' => false]);
        }
        return response()->json(['success' => true]);
    }

    // ── Notices CRUD ───────────────────────────────────────────
    public function storeNotice(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'type' => 'required|in:info,warning,danger,success',
            'is_ticker' => 'boolean',
        ]);

        Notice::create([
            'title' => $request->title,
            'body' => $request->body,
            'type' => $request->type,
            'is_ticker' => $request->boolean('is_ticker'),
            'is_active' => true,
        ]);

        return response()->json(['success' => true]);
    }

    public function updateNotice(Request $request, $id)
    {
        $notice = Notice::findOrFail($id);
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'type' => 'required|in:info,warning,danger,success',
            'is_ticker' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $notice->update($request->only(['title', 'body', 'type', 'is_ticker', 'is_active']));
        return response()->json(['success' => true]);
    }

    public function deleteNotice($id)
    {
        Notice::findOrFail($id)->delete();
        return response()->json(['success' => true]);
    }

    // ── Support Chat ───────────────────────────────────────────
    public function chatIndex($userId)
    {
        $user = User::findOrFail($userId);
        $messages = SupportMessage::where('user_id', $userId)->orderBy('created_at')->get();

        // Mark all user messages as read
        SupportMessage::where('user_id', $userId)->where('sender', 'user')->update(['is_read' => true]);

        return response()->json([
            'user' => ['id' => $user->id, 'name' => $user->name, 'email' => $user->email],
            'messages' => $messages->map(fn(SupportMessage $m) => [
                'id' => $m->id,
                'sender' => $m->sender,
                'message' => $m->message,
                'message_type' => $m->message_type ?? 'text',
                'file_url' => $m->fileUrl(),
                'agent_name' => $m->agent_name,
                'created_at' => $m->created_at->format('d M Y H:i'),
            ]),
        ]);
    }

    public function chatReply(Request $request, $userId)
    {
        $request->validate([
            'message' => 'required|string|max:2000',
            'agent_name' => 'nullable|string|max:100',
        ]);
        User::findOrFail($userId);

        SupportMessage::create([
            'user_id' => $userId,
            'sender' => 'admin',
            'message' => $request->message,
            'message_type' => 'text',
            'agent_name' => $request->agent_name ?: null,
            'is_read' => false,
        ]);

        return response()->json(['success' => true]);
    }

    // ── Agent join / leave (system message) ──────────────────
    public function agentEvent(Request $request, $userId)
    {
        $request->validate([
            'action' => 'required|in:join,leave',
            'agent_name' => 'required|string|max:100',
        ]);
        User::findOrFail($userId);

        $verb = $request->action === 'join' ? 'joined' : 'left';
        $message = $request->agent_name . ' ' . $verb;

        SupportMessage::create([
            'user_id' => $userId,
            'sender' => 'system',
            'message' => $message,
            'message_type' => 'system',
            'agent_name' => $request->agent_name,
            'is_read' => true, // system msgs are always "read"
        ]);

        return response()->json(['success' => true]);
    }

    // ── Admin uploads image/voice in chat ─────────────────────
    public function chatUpload(Request $request, $userId)
    {
        User::findOrFail($userId);
        $type = $request->input('type', 'image');

        if ($type === 'image') {
            $request->validate(['file' => 'required|image|max:5120']);
            $path = $request->file('file')->store('chat/images', 'public');
        } else {
            $request->validate(['file' => 'required|file|mimes:webm,ogg,mp3,wav|max:10240']);
            $ext = $request->file('file')->getClientOriginalExtension() ?: 'webm';
            $name = \Illuminate\Support\Str::uuid() . '.' . $ext;
            $path = $request->file('file')->storeAs('chat/voices', $name, 'public');
        }

        SupportMessage::create([
            'user_id' => $userId,
            'sender' => 'admin',
            'message' => $request->input('caption', '') ?: ($type === 'image' ? '📷 Image' : '🎤 Voice message'),
            'message_type' => $type,
            'file_path' => $path,
            'agent_name' => $request->input('agent_name') ?: null,
            'is_read' => false,
        ]);

        return response()->json(['success' => true]);
    }

    // ── Client management (Subdomains & Files) ────────────────
    public function manageClient($id)
    {
        $user = User::findOrFail($id);
        $userDomains = $this->getUserDomains($user);

        $cpanel = new CPanelService();
        $subdomains = [];
        try {
            $res = $cpanel->listSubdomains();
            $subs = $res['data'] ?? [];
            foreach ($subs as $s) {
                $root = strtolower(trim($s['rootdomain'] ?? ''));
                if (in_array($root, $userDomains, true)) {
                    $subdomains[] = $s;
                }
            }
        } catch (\Throwable $e) {
        }

        return view('admin.manage_client', compact('user', 'userDomains', 'subdomains'));
    }

    public function createClientSubdomain(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $request->validate([
            'subdomain' => 'required|string|regex:/^[a-zA-Z0-9\-]+$/|max:63',
            'root_domain' => 'required|string',
        ]);

        $sub = strtolower(trim($request->subdomain));
        $root = strtolower(trim($request->root_domain));

        if (!in_array($root, $this->getUserDomains($user))) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $cpanel = new CPanelService();
        $res = $cpanel->createSubdomain($sub, $root);

        // Check for error
        if (isset($res['status']) && (int) $res['status'] === 0) {
            $err = $res['errors'][0] ?? 'API error';
            return response()->json(['success' => false, 'message' => $err]);
        }

        // Ensure physical folder exists
        try {
            $cpanel->createFolder("domains/{$root}", $sub);
        } catch (\Throwable $e) {
        }

        // Persist in DB
        UserDomain::updateOrCreate(
            ['domain' => $sub . '.' . $root],
            ['user_id' => $user->id, 'subdomain' => $sub]
        );

        return response()->json([
            'success' => true,
            'full' => $sub . '.' . $root,
            'dir' => "domains/{$root}/{$sub}",
        ]);
    }

    public function deleteClientSubdomain(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $request->validate(['subdomain' => 'required|string']);

        $parts = explode('.', $request->subdomain, 2);
        $root = strtolower(trim($parts[1] ?? ''));

        if (!$root || !in_array($root, $this->getUserDomains($user))) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $cpanel = new CPanelService();
        $res = $cpanel->deleteSubdomain($request->subdomain);

        if (isset($res['cpanelresult']['data'][0]['result']) && (int) $res['cpanelresult']['data'][0]['result'] === 0) {
            $err = $res['cpanelresult']['data'][0]['reason'] ?? 'API error';
            return response()->json(['success' => false, 'message' => $err]);
        }

        // Remove from DB
        UserDomain::where('domain', $request->subdomain)->delete();

        return response()->json(['success' => true]);
    }

    public function clientFiles(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $domain = $request->query('domain');
        $root = $request->query('root', $domain);

        if (!$domain) {
            $userDomains = $this->getUserDomains($user);
            return view('admin.client_file_picker', compact('user', 'userDomains'));
        }

        // Security: Verify domain
        if (!in_array(strtolower($domain), $this->getUserDomains($user))) {
            return redirect()->route('admin.clients.files', $id)->with('error', 'Unauthorized access.');
        }

        $cpanel = new CPanelService();
        $basePath = $root ?: "domains/{$domain}";
        $dir = $request->query('dir', $basePath);

        if (!str_starts_with($dir, $basePath)) {
            $dir = $basePath;
        }

        try {
            $response = $cpanel->listFiles($dir);
            $files = $response['data'] ?? [];
        } catch (\Exception $e) {
            $files = [];
        }

        // Breadcrumbs
        $relativePath = str_replace($basePath, '', $dir);
        $relativePath = ltrim($relativePath, '/');
        $breadcrumbs = [['name' => $domain, 'path' => $basePath]];
        if (!empty($relativePath)) {
            $parts = explode('/', $relativePath);
            $buildPath = $basePath;
            foreach ($parts as $part) {
                $buildPath .= '/' . $part;
                $breadcrumbs[] = ['name' => $part, 'path' => $buildPath];
            }
        }

        return view('admin.client_files', compact('user', 'files', 'domain', 'dir', 'breadcrumbs', 'root'));
    }

    private function getUserDomains(User $user): array
    {
        $domains = [];
        if ($user->primary_domain) {
            $domains[] = strtolower(trim($user->primary_domain));
        }

        $addon = UserDomain::where('user_id', $user->id)
            ->pluck('domain')
            ->map(fn($d) => strtolower(trim($d)))
            ->toArray();

        return array_unique(array_merge($domains, $addon));
    }
}
